<?php declare(strict_types = 1);

namespace Gls\Model;

use Exception;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Doctrine\ORM\ORMException;
use EshopCatalog\FrontModule\Model\Sellers;
use Gls\Model\Entities\GlsOrder;
use Gls\Model\Exception\ParcelGeneration;
use Gls\Model\Exception\ParcelLabels;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Tracy\Debugger;

class OrderApiService
{
	public const LABELS_DIR = TMP_DIR . '/eshopOrders/glsLabels/';

	public function __construct(
		protected EntityManagerDecorator $em,
		protected GlsApi                 $glsApi,
		protected Sellers                $sellers,
	)
    {
    }

	/**
	 * @param GlsOrder[] $orders
	 *
	 * @return int[]
	 */
	public function sendOrders(array $orders, int $quantity = 1): array
	{
		// TODO @roman propsat $quantity do api
		$parcels  = [];
		$api      = $this->glsApi->getApi();
		$orderIds = [];
		$result   = [
			'ok'    => 0,
			'error' => 0,
		];

		foreach ($orders as $glsOrder) {
			$order   = $glsOrder->getOrder();
			$address = $order->getAddressDelivery();
			$seller  = $this->sellers->getSellerForSite($order->site->getIdent());
			$cod     = 0;

			if ($order->getPayment()->getPayment()->getIdent() == 'cod') {
				$cod = round($order->getPrice(true));
			}

			$formattedPhone = str_replace(' ', '', Strings::phoneFormat($address->getPhone(), strtoupper($address->getCountry()->getId())));

			$parcels[] = [
				'SenderName'    => $seller->name,
				'SenderAddress' => $seller->street,
				'SenderCity'    => $seller->city,
				'SenderZipcode' => $seller->postal,
				'SenderCountry' => strtoupper($seller->country->getId()),
				'SenderContact' => $seller->title,
				'SenderPhone'   => str_replace(' ', '', $seller->phone),
				'SenderEmail'   => $seller->email,

				'ConsigName'    => str_replace('&', '&amp;', trim($address->getCompany() ?: $address->getFirstName() . ' ' . $address->getLastName())),
				'ConsigAddress' => $address->getStreet(),
				'ConsigCity'    => $address->getCity(),
				'ConsigZipcode' => $address->getPostal(),
				'ConsigCountry' => strtoupper($address->getCountry()->getId()),
				'ConsigContact' => trim($address->getFirstName() . ' ' . $address->getLastName()),
				'ConsigPhone'   => $formattedPhone,
				'ConsigEmail'   => $address->getEmail(),

				'ClientRef'  => $order->getId(),
				'CodAmount'  => $cod,
				'CodCurr'    => $order->getCurrencyCode(),
				'CodRef'     => (string) $order->getId(),
				'Pcount'     => 1,
				'PickupDate' => date("Y-m-d"),
				'Services'   => [
					"FDS" => $address->getEmail(),
					"FSS" => $formattedPhone,
				],
			];

			$orderIds[$order->getId()] = $order->getId();
		}

		$parcelNumbers = $api->getParcelNumbers($parcels);
		foreach ($parcelNumbers as $orderId => $number) {
			/** @var GlsOrder $entity */
			$entity                = $this->em->getReference(GlsOrder::class, $orderId);
			$entity->numberPackage = $number;
			$entity->export();
			$this->em->persist($entity);
			unset($orderIds[$orderId]);
			$result['ok']++;
		}
		$this->em->flush();

		$result['error'] = count($orderIds);

		return $result;
	}

	/**
     * @param GlsOrder[] $orders
     *
     * @throws ParcelLabels
     */
    public function generateLabels(array $orders): array
	{
		$numbers = array_map(static fn(GlsOrder $order) => $order->numberPackage, $orders);
		foreach ($orders as $o) {
			foreach ($o->getAssociatedNumberPackages() as $anp) {
				$numbers[] = $anp->getNumberPackage();
			}
		}
		// TODO @roman prekontrolovat, ze se generuji stitky pro VK zasilky (tzn. pro cisla zasilek z getAssociatedNumberPackages())
		$api    = $this->glsApi->getApi();
		$result = [
			'ok'    => 0,
			'error' => 0,
			'files' => [],
		];

		$label = $api->getParcelLabels($numbers);

		if ($label['status'] == 'success') {
			$file = self::LABELS_DIR . time() . '_' . uniqid() . '.pdf';
			FileSystem::createDir(dirname($file));
			$pdf = fopen($file, 'w');
			fwrite($pdf, (string) $label['pdf']);
			fclose($pdf);
			$result['files'][] = $file;
			$result['ok']++;
		} else {
			Debugger::log($label['error_description'], 'gls/_labels');
			$result['error']++;
		}

		// Odstraneni starych stitku
		$today = (new DateTime())->format('Y-m-d');
		foreach (glob(self::LABELS_DIR . '*.pdf') as $file) {
			$time = explode('_', basename($file))[0];

			if (Strings::isValidTimestamp((int) $time)) {
				$date = DateTime::from($time)->format('Y-m-d');

				if ($date < $today) {
					unlink($file);
				}
			}
		}

		return $result;
	}

	public function checkCompleted(GlsOrder $glsOrder): bool
	{
		$api = $this->glsApi->getApi();

		try {
			$result = $api->getParcelStatusList($glsOrder->numberPackage);

			if ($result === false) {
				$glsOrder->lastStatus = GlsOrder::STATUS_NOT_FOUND;
				$this->em->persist($glsOrder);
				$this->em->flush();

				return false;
			}

			foreach ($result as $status) {
				if ($status['StCode'] == '23') {
					$glsOrder->lastStatus = GlsOrder::STATUS_RETURNED;
					$this->em->persist($glsOrder);
					$this->em->flush();

					return false;
				}
			}

			if (isset($result[0]['StCode']) && $result[0]['StCode'] == '5') {
				return true;
			}
		} catch (Exception) {
			Debugger::log("Cannot check complete data for order '{$glsOrder->getOrder()->getId()}' ({$glsOrder->numberPackage}) - "
				. $api->getTrackingUrlXml($glsOrder->numberPackage), 'gls');
		}

		return false;
	}

	public function getTrackingUrlXml(string $packageNumber): string
	{
		return $this->glsApi->getApi()->getTrackingUrlXml($packageNumber);
	}
}
