<?php declare(strict_types = 1);

namespace Gls\Model;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Core\Model\Helpers\BaseEntityService;
use Gls\Model\Entities\GlsOrder;

/**
 * @method GlsOrder|null getReference($id)
 * @method GlsOrder[] getAll()
 * @method GlsOrder|null get($id)
 */
class OrdersExported extends BaseEntityService
{
	protected $entityClass = GlsOrder::class;

	/**
	 * @param int[]|string[] $ids
	 * @return GlsOrder[]
	 */
	public function getOrdersNotExported(array $ids = []): array
	{
		return $this->getOrders(false, $ids);
	}

	/**
	 * @return GlsOrder[]
	 */
	public function getOrdersExported(array $ids = [], bool $loadFull = true): array
	{
		return $this->getOrders(true, $ids, $loadFull);
	}

	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ($this->getEr()->createQueryBuilder('go')
			         ->where('go.lastStatus NOT IN (:status) OR go.lastStatus IS NULL')
			         ->andWhere('go.numberPackage IS NOT NULL')
			         ->setParameters(new ArrayCollection([new Parameter('status', [
					         GlsOrder::STATUS_COMPLETED,
					         GlsOrder::STATUS_NOT_FOUND,
					         GlsOrder::STATUS_RETURNED,
				         ])]))->getQuery()
			         ->getResult() as $row) {
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	/**
	 * @param int[]|string[] $ids
	 * @return GlsOrder[]
	 */
	public function getOrders(?bool $isExported = null, array $ids = [], bool $loadFull = true): array
	{
		$qb = $this->getEr()->createQueryBuilder('go')
			->addSelect('anp')
			->leftJoin('go.associatedNumberPackages', 'anp')
			->orderBy('go.order')->groupBy('go.order');

		if ($loadFull) {
			$qb->addSelect('o, go, oad, oai, os, s')
				->join('go.order', 'o')
				->leftJoin('o.addressDelivery', 'oad')
				->leftJoin('o.addressInvoice', 'oai')
				->leftJoin('o.spedition', 'os')
				->leftJoin('os.spedition', 's');
		}

		if ($isExported === true) {
            $qb->where('go.exported IS NOT NULL');
        } elseif ($isExported === false) {
            $qb->where('go.exported IS NULL');
        }

		if ($ids !== []) {
			$qb->andWhere('go.order IN (:ids)')->setParameter('ids', $ids);
		}

		$result = [];
		foreach ($qb->getQuery()->getResult() as $item) {
			$result[$item->getOrder()->getId()] = $item;
		}

		return $result;
	}

	public function markAsCompleted(array $orderIds): void
	{
		foreach ($this->getEr()->createQueryBuilder('go')
			         ->where('go.order IN (:ids)')
			         ->setParameter('ids', $orderIds)
			         ->getQuery()->getResult() as $row) {
			/** @var GlsOrder $row */
			$row->lastStatus = GlsOrder::STATUS_COMPLETED;
			$this->em->persist($row);
		}

		$this->em->flush();
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];
		foreach ($this->getEr()->createQueryBuilder('g')
			         ->select('IDENTITY(g.order) as id')
			         ->leftJoin('g.associatedNumberPackages', 'acp')
			         ->orWhere('g.numberPackage LIKE :number')
			         ->orWhere('acp.numberPackage LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}
}
