<?php declare(strict_types = 1);

namespace Google\Model\Reviews;

use Core\Model\Event\Event;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\BaseFrontEntityService;
use Core\Model\Helpers\Strings;
use DateTime;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Exception;
use Google\FrontModule\Model\GoogleCache;
use Google\Model\Entities\GoogleReview;
use Google\Model\Events\GoogleReviewEvent;
use Nette\Utils\Json;
use Tracy\Debugger;

/**
 * @method GoogleReview|null getReference($id)
 */
class ReviewsService extends BaseFrontEntityService
{
	protected $entityClass = GoogleReview::class;

	public function __construct(
		protected EventDispatcher $eventDispatcher,
		protected GoogleCache     $googleCache,
	)
	{
	}

	public function downloadRating(string $key, string $apiKey, string $placeId): void
	{
		$curl = curl_init();

		curl_setopt_array($curl, [
			CURLOPT_URL            => 'https://maps.googleapis.com/maps/api/place/details/json?place_id=' . $placeId . '&fields=rating%2Cuser_ratings_total%2Creviews&key=' . $apiKey,
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING       => '',
			CURLOPT_MAXREDIRS      => 10,
			CURLOPT_TIMEOUT        => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST  => 'GET',
		]);

		$response = curl_exec($curl);

		curl_close($curl);

		try {
			if ($response && Arrays::isJson((string) $response)) {
				$data = ((object) Json::decode((string) $response))->result ?? null;

				if (!$data) {
					return;
				}

				$conn = $this->em->getConnection();

				$exist = $conn->fetchOne("SELECT id FROM google__location_rating WHERE id = ?", [$key]);
				if ($exist) {
					$conn->update('google__location_rating', [
						'rating'        => $data->rating,
						'reviews_count' => $data->user_ratings_total,
						'updated'       => (new DateTime())->format('Y-m-d H:i:s'),
					], [
						'id' => $key,
					]);
				} else {
					$conn->insert('google__location_rating', [
						'id'            => $key,
						'rating'        => $data->rating,
						'reviews_count' => $data->user_ratings_total,
						'updated'       => (new DateTime())->format('Y-m-d H:i:s'),
					]);
				}

				$this->googleCache->clearLocationRating($key);
			}
		} catch (Exception $e) {
			Debugger::log($e);
		}
	}

	public function downloadForPlace(string $apiKey, string $placeId): void
	{
		$current = [];

		foreach ($this->getEr()->createQueryBuilder('r')
			         ->select('r.id, r.reviewId')
			         ->where('r.placeId = :placeId')
			         ->setParameters(new ArrayCollection([new Parameter('placeId', $placeId)]))->getQuery()->getArrayResult() as $row) {
			/** @var array $row */
			$current[$row['reviewId']] = $row['id'];
		}

		$curl = curl_init();

		curl_setopt_array($curl, [
			CURLOPT_URL            => 'https://maps.googleapis.com/maps/api/place/details/json?place_id=' . $placeId . '&fields=place_id%2Creviews%2Crating&key=' . $apiKey . '&reviews_sort=newest&reviews_no_translations=true',
			CURLOPT_RETURNTRANSFER => true,
			CURLOPT_ENCODING       => '',
			CURLOPT_MAXREDIRS      => 10,
			CURLOPT_TIMEOUT        => 0,
			CURLOPT_FOLLOWLOCATION => true,
			CURLOPT_HTTP_VERSION   => CURL_HTTP_VERSION_1_1,
			CURLOPT_CUSTOMREQUEST  => 'GET',
		]);

		$response = curl_exec($curl);

		curl_close($curl);

		try {
			if ($response && Arrays::isJson((string) $response)) {

				$data = ((object) Json::decode((string) $response))->result;

				$this->eventDispatcher->dispatch(new Event((array) $data), 'google.googlePlace::dataDownload');

				foreach ($data->reviews as $row) {
					$reviewId = md5($placeId . $row->author_name . $row->time);

					if (!isset($current[$reviewId])) {
						$text = $row->text ? Strings::emojiToHtmlEntity((string) $row->text) : null;

						$review                  = new GoogleReview;
						$review->placeId         = $placeId;
						$review->authorName      = $row->author_name;
						$review->authorUrl       = $row->author_url;
						$review->profilePhotoUrl = $row->profile_photo_url;
						$review->rating          = (string) $row->rating;
						$review->text            = is_string($text) ? $text : null;
						$review->created         = date('Y-m-d H:i:s', $row->time);
						$review->language        = $row->language;
						$review->reviewId        = $reviewId;

						$this->em->persist($review);
						$this->em->flush();

						$event               = new GoogleReviewEvent($reviewId);
						$event->googleReview = $review;
						$this->eventDispatcher->dispatch($event, 'google.googleReview::created');
					}
				}
			}
		} catch (Exception $e) {
			Debugger::log($e);
		}
	}

}
