<?php declare(strict_types=1);

namespace GoPay\FrontModule\Components;

use Closure;
use Contributte\GopayInline\Api\Entity\PaymentFactory;
use Contributte\GopayInline\Exception\GopayException;
use Contributte\GopayInline\Http\Response;
use Core\Model\UI\BaseControl;
use GoPay\FrontModule\Model\Gopay;
use Nette\Application\UI\Presenter;
use Nette\Http\Session;
use Nette\Http\SessionSection;

class GopayControl extends BaseControl
{
	protected array $payment;
	protected Gopay $client;
	protected SessionSection $sessionSection;

	/** @var Closure[] */
	public array $onCheckout = [];

	/** @var Closure[] */
	public array $onSuccess = [];

	/** @var Closure[] */
	public array $onError = [];

	public function __construct(array $payment, Gopay $client, Session $session)
	{
		$this->payment = $payment;
		$this->client = $client;
		$this->sessionSection = $session->getSection('gopay');
	}

	public function render(array $attrs = [], string $text = 'Pay'): void
	{
		$template = $this->template;
		$template->setFile($this->getTemplateFile());
		$template->add('checkoutLink', $this->link('//checkout!'));
		$template->add('attrs', $attrs);
		$template->add('text', $text);
		$template->render();
	}

	public function handleCheckout(): void
	{
		$response = null;

		try {
			/** @var Presenter $presenter */
			$presenter = $this->getPresenter();

			$paymentFinishedUrl = $presenter->link('//:EshopOrders:Front:Payment:paymentFinished', ['orderIdent' => $presenter->getParameter('orderIdent'), 'do' => 'goPayButton-success']);
			$this->payment['callback']['return_url'] = $paymentFinishedUrl;
			$this->payment['callback']['notify_url'] = $paymentFinishedUrl;
			$response = $this->client->payments->createPayment(PaymentFactory::create($this->payment));

			if (!$response->isSuccess()) {
				$this->errorHandler(new GopayException($response->getError(), $response->getCode()));
			}

			$data = $response->getData();

			/** @phpstan-ignore-next-line */
			$url = $response->gw_url;

			/** @phpstan-ignore-next-line */
			$this->sessionSection->payment_id = $response->id;
			$this->onCheckout($this, $this->payment, $data);

			$presenter->redirectUrl($url);

		} catch (GopayException $exception) {
			$this->errorHandler($exception, $response);
		}
	}

	public function handleSuccess(): void
	{
		$paymentId = $this->sessionSection->payment_id;
		$this->sessionSection->remove();

		try {
			$verify = $this->client->payments->verify($paymentId);

			if (in_array($verify->getData()['state'], ['CANCELED', 'TIMEOUTED'])) {
				$this->errorHandler(new GopayException($verify->getData()['state'], $verify->getCode()), $verify);
			} else {
				$this->onSuccess($this, $verify);
			}

		} catch (GopayException $exception) {
			$this->errorHandler($exception);
		}
	}

	public function errorHandler(GopayException $exception, ?Response $response = null): void
	{
		if (!$this->onError) {
			throw $exception;
		}

		$this->onError($this, $exception, $response);
	}
	
}
