<?php declare(strict_types = 1);

namespace GoPay\FrontModule\Model\Subscribers;

use Contributte\GopayInline\Api\Lists\Language;
use Contributte\GopayInline\Exception\GopayException;
use Contributte\Translation\Translator;
use Core\Model\Event\PresenterTemplateEvent;
use Core\Model\Helpers\Strings;
use Core\Model\UI\BaseControl;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\FrontModule\Presenters\FinishedPresenter;
use EshopOrders\FrontModule\Presenters\PaymentPresenter;
use EshopOrders\Model\Entities\OrderCardPaymentToken;
use EshopOrders\Model\Orders;
use Exception;
use GoPay\FrontModule\Components\GopayControl;
use GoPay\FrontModule\Components\IGopayControlFactory;
use GoPay\FrontModule\Model\Gopay;
use Money\Money;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Throwable;
use Tracy\Debugger;

class OrderPaymentSubscriber implements EventSubscriberInterface
{
	protected IGopayControlFactory $gopayControlFactory;
	protected CardsPaymentService  $cardsPaymentService;
	protected Orders               $orders;
	protected Translator           $translator;
	protected Gopay                $client;

	public function __construct(IGopayControlFactory $gopayControlFactory, CardsPaymentService $cardsPaymentService, Orders $orders, Translator $translator, Gopay $client)
	{
		$this->gopayControlFactory = $gopayControlFactory;
		$this->cardsPaymentService = $cardsPaymentService;
		$this->orders              = $orders;
		$this->translator          = $translator;
		$this->client              = $client;
	}

	/**
	 * @return string[]
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderFinishedRender'   => 'orderFinishedRender',
			'eshopOrders.paymentPayAction'      => 'paymentPayAction',
			'eshopOrders.paymentFinishedRender' => 'paymentFinishedRender',
		];
	}

	public function orderFinishedRender(PresenterTemplateEvent $event): void
	{
		/** @var FinishedPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;

		if (!$order || $order->getPayment()->getPayment()->getIdent() !== 'card') {
			return;
		}

		$presenter->template->link = $presenter->link('Payment:pay', ['orderIdent' => $order->getIdent()]);
		$presenter->addIncludeTemplate(__DIR__ . '/GopayButton.latte');
	}

	public function paymentPayAction(PresenterTemplateEvent $event): void
	{
		/** @var PaymentPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;

		$token    = $this->cardsPaymentService->createToken($order);
		$customer = $order->getAddressInvoice();

		$items = [];
		foreach ($order->getOrderItems()->toArray() as $orderItem) {
			$items[] = [
				'name'   => $orderItem->getOrderItemText()->getName(),
				'amount' => Money::{$order->getCurrencyCode()}(round($orderItem->getPriceTotal(), 2) * 100),
			];
		}

		if ($order->getSpedition()->getPrice()) {
			$items[] = [
				'name'   => $order->getSpedition()->getName(),
				'amount' => Money::{$order->getCurrencyCode()}(round($order->getSpedition()->getPrice(), 2) * 100),
			];
		}

		if ($order->getPayment()->getPrice()) {
			$items[] = [
				'name'   => $order->getPayment()->getName(),
				'amount' => Money::{$order->getCurrencyCode()}(round($order->getPayment()->getPrice(), 2) * 100),
			];
		}

		$payment = [
			'payer'        => [
				'contact' => [
					'first_name'   => $customer->getFirstName(),
					'last_name'    => $customer->getLastName(),
					'email'        => $customer->getEmail(),
					'phone_number' => str_replace(' ', '', Strings::phoneFormat($customer->getPhone())),
					'city'         => $customer->getCity(),
					'street'       => $customer->getStreet(),
					'postal_code'  => $customer->getPostal(),
					'country_code' => $customer->getCountry() ? $customer->getCountry()->getIso3166_1() : 'CZE',
				],
			],
			'amount'       => Money::{$order->getCurrencyCode()}(round($order->getPrice(true), 2) * 100),
			'order_number' => (string) $order->getId(),
			'items'        => $items,
			'lang'         => $this->translator->getLocale() === 'cs' ? Language::CZ : Language::EN,
		];

		$control = $this->gopayControlFactory->create($payment);

		$presenter->addComponent($control, 'goPayButton');

		$control->onCheckout[] = function(GopayControl $control, array $payment, $data) use ($token) {
			$this->cardsPaymentService->tokenCheckout($token->getId());
		};

		$control->handleCheckout();
	}

	public function paymentFinishedRender(PresenterTemplateEvent $event): void
	{
		/** @var PaymentPresenter $presenter */
		$presenter  = $event->presenter;
		$params     = $presenter->getParameters();
		$orderIdent = $params['orderIdent'];
		$paymentId  = $params['id'];

		if ($paymentId) {
			$order = $this->orders->getByIdent($orderIdent);

			if (!$order || $order->getPaymentIdent() !== 'card') {
				return;
			}

			$presenter->order           = $order;
			$presenter->template->order = $order;
			$token                      = $this->cardsPaymentService->getLastCreated($order->getId());

			if ($token && !$token->param) {
				$token->param = $paymentId;
				$this->orders->em->persist($token);
				$this->orders->em->flush($token);
			}

			try {
				$verify = $this->client->payments->verify($paymentId);

				if (in_array($verify->getData()['state'], ['CANCELED', 'TIMEOUTED'])) {
					$msg = $verify->getData()['state'];
					$this->errorHandler(new GopayException($msg, $verify->getCode()), $token);
					$this->cardsPaymentService->tokenError($token->getId(), $msg);
					$presenter->template->cardError = $msg;
				} else if ($token && $verify->getData()['state'] == 'PAID') {
					$this->cardsPaymentService->tokenPaid($token->getToken());
				} else {
					$presenter->template->cardError = $verify->getData()['state'];
				}
			} catch (Exception $e) {
				Debugger::log($e);

				if ($token) {
					$this->errorHandler($e, $token);
				}
			}
		}

		$presenter->addComponent($this->createFakeComponent(), 'goPayButton');
		$presenter->addIncludeTemplate(__DIR__ . '/PaymentFinished.latte');
	}

	protected function errorHandler(Throwable $ex, OrderCardPaymentToken $token): void
	{
		Debugger::log($ex->getMessage(), 'gopay');
		$this->cardsPaymentService->tokenError($token->getId(), $ex->getMessage());
	}

	protected function createFakeComponent(): BaseControl
	{
		return new class extends BaseControl {
			public function render(): void { }

			public function handleSuccess(): void { }

			public function handleNotify(): void { }
		};
	}

}
