<?php declare(strict_types = 1);

namespace GPWebPay\FrontModule\Model\Subscribers;

use Contributte\Translation\Translator;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\PresenterTemplateEvent;
use Core\Model\Helpers\Strings;
use EshopOrders\FrontModule\Model\CardsPaymentService;
use EshopOrders\FrontModule\Presenters\FinishedPresenter;
use EshopOrders\FrontModule\Presenters\PaymentPresenter;
use Pixidos\GPWebPay\Components\GPWebPayControl;
use Pixidos\GPWebPay\Components\GPWebPayControlFactory;
use Pixidos\GPWebPay\Data\Operation;
use Pixidos\GPWebPay\Data\RequestInterface;
use Pixidos\GPWebPay\Data\ResponseInterface;
use Pixidos\GPWebPay\Enum\Currency as CurrencyEnum;
use Pixidos\GPWebPay\Exceptions\GPWebPayException;
use Pixidos\GPWebPay\Factory\ResponseFactory;
use Pixidos\GPWebPay\Param\AddInfo;
use Pixidos\GPWebPay\Param\Amount;
use Pixidos\GPWebPay\Param\Currency;
use Pixidos\GPWebPay\Param\MerOrderNum;
use Pixidos\GPWebPay\Param\OrderNumber;
use Pixidos\GPWebPay\Param\ResponseUrl;
use Pixidos\GPWebPay\ResponseProviderInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Tracy\Debugger;

class OrderPaymentSubscriber implements EventSubscriberInterface
{
	protected GPWebPayControlFactory    $controlFactory;
	protected ResponseProviderInterface $responseProvider;
	protected ResponseFactory           $responseFactory;
	protected CardsPaymentService       $cardsPaymentService;
	protected Translator                $translator;
	protected EntityManagerDecorator    $em;

	public function __construct(
		GPWebPayControlFactory    $controlFactory,
		ResponseProviderInterface $responseProvider,
		ResponseFactory           $responseFactory,
		CardsPaymentService       $cardsPaymentService,
		Translator                $translator,
		EntityManagerDecorator    $em
	)
	{
		$this->controlFactory      = $controlFactory;
		$this->responseProvider    = $responseProvider;
		$this->responseFactory     = $responseFactory;
		$this->cardsPaymentService = $cardsPaymentService;
		$this->translator          = $translator;
		$this->em                  = $em;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			'eshopOrders.orderFinishedRender'   => 'orderFinishedRender',
			'eshopOrders.paymentPayAction'      => 'paymentPayAction',
			'eshopOrders.paymentFinishedRender' => 'paymentFinishedRender',
		];
	}

	public function orderFinishedRender(PresenterTemplateEvent $event): void
	{
		/** @var FinishedPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;

		if (!$order || $order->getPayment()->getPayment()->getIdent() !== 'card') {
			return;
		}

		$presenter->template->link = $presenter->link('Payment:pay', ['orderIdent' => $order->getIdent()]);
		$presenter->addIncludeTemplate(__DIR__ . '/GPWebPayButton.latte');
	}

	public function paymentPayAction(PresenterTemplateEvent $event): void
	{
		/** @var PaymentPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;
		$self      = $this;

		if ($order->getPaymentIdent() !== 'card') {
			return;
		}

		$token = $this->cardsPaymentService->createToken($order);

		$operation = new Operation(
			new OrderNumber($token->getToken()),
			new Amount($order->getPrice(true)),
			new Currency(CurrencyEnum::{$order->getCurrencyCode()}()),
			null,
			new ResponseUrl($presenter->link('//:EshopOrders:Front:Payment:paymentFinished', ['orderIdent' => $order->getIdent()])),
		);
		$operation->addParam(new MerOrderNum($order->getId()));

		$invAddr  = $order->getAddressInvoiceRaw();
		$deliAddr = $order->getAddressDeliveryRaw();

		if (!$invAddr && !$deliAddr) {
			return;
		}

		$formattedPhone = str_replace(' ', '', Strings::phoneFormat($invAddr->getPhone(), strtoupper($invAddr->getCountry()->getId())));

		$formattedPhone = ltrim($formattedPhone, '+');
		$phonePrefix    = mb_substr($formattedPhone, -12, 3);
		$formattedPhone = mb_substr($formattedPhone, -9);

		$infoXml = '
<additionalInfoRequest version="4.0" xmlns="http://gpe.cz/gpwebpay/additionalInfo/request">
  <cardholderInfo>
    <cardholderDetails>
      <name>' . Strings::truncate((string) $invAddr->getName(), 45, '') . '</name>
      <email>' . Strings::truncate((string) $invAddr->getEmail(), 255, '') . '</email>
      <mobilePhoneCountry>' . $phonePrefix . '</mobilePhoneCountry>
      <mobilePhone>' . $formattedPhone . '</mobilePhone>
    </cardholderDetails>
    ';

		if (!$deliAddr) {
			$infoXml .= '<addressMatch>Y</addressMatch>
      <billingDetails>
        <name>' . Strings::truncate((string) $invAddr->getName(), 255, '') . '</name>
        <address1>' . Strings::truncate((string) $invAddr->getStreet(), 50, '') . '</address1>
        <city>' . Strings::truncate((string) $invAddr->getCity(), 50, '') . '</city>
        <postalCode>' . Strings::truncate((string) $invAddr->getPostal(), 16, '') . '</postalCode>
        <country>' . $this->getCountryIsoNumber($invAddr->getCountry()->getId()) . '</country>
      </billingDetails>
      ';
		} else {
			$infoXml .= '
	<addressMatch>N</addressMatch>
      <billingDetails>
        <name>' . Strings::truncate((string) $invAddr->getName(), 255, '') . '</name>
        <address1>' . Strings::truncate((string) $invAddr->getStreet(), 50, '') . '</address1>
        <city>' . Strings::truncate((string) $invAddr->getCity(), 50, '') . '</city>
        <postalCode>' . Strings::truncate((string) $invAddr->getPostal(), 16, '') . '</postalCode>
        <country>' . $this->getCountryIsoNumber($invAddr->getCountry()->getId()) . '</country>
      </billingDetails>
      <shippingDetails>
        <name>' . Strings::truncate((string) $deliAddr->getName(), 255, '') . '</name>
        <address1>' . Strings::truncate((string) $deliAddr->getStreet(), 50, '') . '</address1>
        <city>' . Strings::truncate((string) $deliAddr->getCity(), 50, '') . '</city>
        <postalCode>' . Strings::truncate((string) $deliAddr->getPostal(), 16, '') . '</postalCode>
        <country>' . $this->getCountryIsoNumber($deliAddr->getCountry()->getId()) . '</country>
      </shippingDetails>
      ';
		}

		$infoXml .= '</cardholderInfo>
</additionalInfoRequest>';

		$infoXml = '<' . '?xml version="1.0" encoding="UTF-8"?>' . trim($infoXml);
		$infoXml = preg_replace('/>\s+?</', '><', $infoXml);

		$operation->addParam(new AddInfo($infoXml));

		$control = $this->controlFactory->create($operation);

		$presenter->addComponent($control, 'webPayButton');

		$control->onCheckout[] = static function(GPWebPayControl $control, RequestInterface $request) use ($self) {
			$self->cardsPaymentService->tokenCheckout((int) $request->getParams()['ORDERNUMBER']);
		};

		$control->handleCheckout();
	}

	public function paymentFinishedRender(PresenterTemplateEvent $event): void
	{
		/** @var PaymentPresenter $presenter */
		$presenter = $event->presenter;
		$order     = $presenter->order;
		$self      = $this;

		if (!$order || $order->getPayment()->getPayment()->getIdent() !== 'card') {
			return;
		}

		$response = $this->responseFactory->create($presenter->getParameters());
		$this->responseProvider->addOnSuccess(
			static function(ResponseInterface $response) use ($self) {
				$self->cardsPaymentService->tokenPaid((int) $response->getOrderNumber());
			}
		);

		$this->responseProvider->addOnError(
			static function(GPWebPayException $exception, ResponseInterface $response) use ($presenter, $self) {
				Debugger::log($exception, 'gpwebpay');
				$self->cardsPaymentService->tokenError((int) $response->getOrderNumber(), $response->getResultText());
				$presenter->template->cardError = $response->getResultText();
				$presenter->template->payLink   = $presenter->link('Payment:pay', ['orderIdent' => $presenter->order->getIdent()]);
			}
		);

		$presenter->addIncludeTemplate(__DIR__ . '/PaymentFinished.latte');

		$this->responseProvider->provide($response);
	}

	public function getCountryIsoNumber(string $country): int
	{
		$country = Strings::upper($country);
		switch ($country) {
			case 'CZ':
				return 203;
			case 'SK':
				return 703;
			case 'PL':
				return 616;
			case 'DE':
				return 276;
		}

		return 203;
	}
}
