<?php declare(strict_types = 1);

namespace Import\AdminModule\Components;

use Core\Components\Flashes\Flashes;
use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Import\AdminModule\Model\Imports;
use Import\Model\Entities\Import;
use Import\Model\Entities\ImportHistory;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;

class ImportsGrid extends BaseControl
{
	/** @var Imports */
	protected $importsService;

	public function __construct(Imports $imports)
	{
		$this->importsService = $imports;
	}

	public function render()
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handleRun($id)
	{
		bdump($id);
	}

	public function handleDelete($id)
	{
		$presenter = $this->getPresenter();

		if (!$presenter->getUser()->isAllowed('Import:Admin', 'create'))
			throw new ForbiddenRequestException();

		if ($this->importsService->remove($id))
			$presenter->flashMessageSuccess('default.deleted');
		else
			$presenter->flashMessageDanger('default.removeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid()
	{
		$grid = $this->createGrid();

		$qb = $this->importsService->getEr()->createQueryBuilder('i')
			->select('i, h')
			->leftJoin('i.history', 'h')
			->orderBy('i.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('importsGrid:gridSortableRow!');

		// Columns
		$grid->addColumnText('title', 'import.import.title')->setRenderer(function($row) {
			/** @var Import $row */
			if ($this->getPresenter()->getUser()->isAllowed('Import:Admin', 'create')) {
				return Html::el('a', ['href' => $this->getPresenter()->link('Default:edit', $row->getId())])
					->setText($row->title);
			} else {
				return $row->title;
			}
		});
		$grid->addColumnStatus('isActive', 'default.isActive')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];
		$grid->addColumnText('type', 'import.import.type');
		$grid->addColumnText('lastRun', 'import.import.lastRun')->setRenderer(function($row) {
			/** @var Import $row */
			if ($row->getHistory()->first()) {
				/** @var ImportHistory $h */
				$h   = $row->getHistory()->first();
				$msg = '';
				foreach ($h->data as $k => $v)
					$msg .= $k . ' -> ' . $v . '<br>';

				return Html::el()
					->addHtml(Html::el('div')->setText($h->created->format('Y-m-d H-i-s')))
					->addHtml(Html::el('dic .text-muted')->setHtml($msg));
			}

			return '';
		});

		// Actions
		$grid->addAction('run', '', ':Import:Cron:Default:run')->setIcon('fas fa-play')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('fas fa-times')->setBsType('danger')
			->setRenderCondition(function($row) { return $this->getPresenter()->getUser()->isAllowed('Import:Admin', 'create'); });

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	public function gridPublishChange($id, $newStatus)
	{
		$presenter = $this->getPresenter();

		if ($this->importsService->setActive($id, $newStatus))
			$presenter->flashMessageSuccess('default.publishChanged');
		else
			$presenter->flashMessageDanger('default.publishChangeFailed');

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else
			$presenter->redirect('this');
	}

	public function handleGridSortableRow()
	{
		$presenter = $this->getPresenter();
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id', null);
		$position  = $request->getPost('position', null);

		if ($id != null && $position != null && $this->importsService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}
}
