<?php declare(strict_types = 1);

namespace Import\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TActive;
use Import\AdminModule\Model\Readers\XmlReader;
use Import\Model\Entities\Import;
use Import\Model\Entities\ImportHistory;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Strings;

/**
 * Class Imports
 * @package Import\AdminModule\Model
 *
 * @method Import|object|null getReference($id)
 * @method Import[]|null getAll()
 * @method Import|null get($id)
 */
class Imports extends BaseEntityService
{
	use TActive;

	protected $entityClass = Import::class;

	/** @var array */
	protected $fileContent = [];

	/** @var array */
	protected $cRemoteLastUpdate = [];

	/**
	 * @param Import $import
	 *
	 * @return mixed|null
	 */
	public function getFileContent(Import $import)
	{
		if (!isset($this->fileContent[$import->getId()]) && $import->url) {
			$file = $this->downloadFile($import->url);

			$result = [
				'contentType' => mime_content_type($file),
				'body'        => $file ? @file_get_contents($file) : '',
			];

			$this->fileContent[$import->getId()] = $result;
		}

		return $this->fileContent[$import->getId()] ?? null;
	}

	/**
	 * @param string $fileContent
	 * @param Import $import
	 *
	 * @return array
	 */
	public function getFileData($fileContent, $import)
	{
		$importFileData = ['data' => [], 'keys' => [], 'values' => []];

		switch ($fileContent['contentType']) {
			case 'text/xml':
				$reader = new XmlReader($fileContent['body']);
				if ($import->getSyncOpt('xPath'))
					$reader->setXpath($import->getSyncOpt('xPath'));

				// TODO možná rozdělit
				$importFileData['data']   = $reader->getData();
				$importFileData['keys']   = $reader->getKeys();
				$importFileData['values'] = $reader->getValues();
				break;
		}

		return $importFileData;
	}

	/**
	 * @param string $url
	 *
	 * @return int
	 */
	public function remoteLastUpdate($url)
	{
		if (!isset($this->cRemoteLastUpdate[$url])) {
			$curl = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_NOBODY, true);
			curl_setopt($curl, CURLOPT_HEADER, true);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_FILETIME, true);

			$result = curl_exec($curl);
			$info   = curl_getinfo($curl);

			curl_close($curl);

			$this->cRemoteLastUpdate[$url] = $info['filetime'];
		}

		return $this->cRemoteLastUpdate[$url];
	}

	/**
	 * @param string $url
	 *
	 * @return bool
	 */
	public function needDownload($url)
	{
		$remoteLastUpdate = $this->remoteLastUpdate($url);
		if ($remoteLastUpdate == -1)
			return true;

		return file_exists($this->getTmpFile($url, $remoteLastUpdate)) ? false : true;
	}

	/**
	 * @param Import $import
	 *
	 * @return bool
	 */
	public function needUpdate($import)
	{
		return $import->url && $import->lastRun ? (file_exists($this->getTmpFile($import->url, $import->lastRun->getTimestamp())) ? false : true) : true;
	}

	/**
	 * @param Import $import
	 *
	 * @throws \Exception
	 */
	public function saveInfoNoNewData($import)
	{
		$history       = new ImportHistory($import, 'ok');
		$history->data = ['message' => 'No new data'];
		$this->em->persist($history);
		$this->em->flush();
	}

	/**
	 * @param Import $import
	 *
	 * @throws \Exception
	 */
	public function updateLastRun($import)
	{
		$import->lastRun = DateTime::from($this->remoteLastUpdate($import->url));
		$this->em->persist($import)->flush();
	}

	/**
	 * @param string $url
	 * @param int    $modified
	 *
	 * @return string
	 */
	protected function getTmpFile($url, $modified) { return TMP_DIR . '/imports/' . Strings::webalize($url) . '__m' . $modified; }

	/**
	 * Stáhne se soubor podle importu podle data modifikace
	 *
	 * @param string $url
	 *
	 * @return string
	 */
	protected function downloadFile($url)
	{
		$tmpFile = $this->getTmpFile($url, $this->remoteLastUpdate($url));
		if ($this->needDownload($url)) {
			foreach (glob(TMP_DIR . '/imports/' . Strings::webalize($url) . '*') as $v)
				unlink($v);
			FileSystem::copy($url, $tmpFile);
		}

		return $tmpFile;
	}

	/**
	 * @param int $id
	 * @param int $position
	 *
	 * @return bool
	 * @throws \Exception
	 */
	public function setPosition($id, $position)
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	/**
	 * Clean for memory usage
	 */
	public function clean()
	{
		$this->cRemoteLastUpdate = [];
		$this->fileContent       = [];
	}
}
