<?php declare(strict_types = 1);

namespace Import\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Exception;
use Import\AdminModule\Model\DataDownloaders;
use Import\AdminModule\Model\Imports;
use Import\AdminModule\Model\ImportType\ImportType;
use Import\AdminModule\Model\ImportTypes;
use Import\AdminModule\Model\ProcessingTypes;
use Import\Model\Entities\Import;
use Import\Model\ImportConfig;
use Nette\Application\BadRequestException;
use Nette\Utils\ArrayHash;

class ImportForm extends BaseControl
{
	protected ?Import $import = null;

	public function __construct(
		protected Imports         $importsService,
		protected ImportTypes     $importTypes,
		protected ProcessingTypes $processingTypes,
		protected DataDownloaders $dataDownloaders,
	)
	{
	}

	public function render(): void
	{
		$this->template->import = $this->import;
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('title', 'import.import.title');
		$form->addTextArea('description', 'import.import.description');
		$form->addBool('isActive', 'import.import.isActive');
		$form->addText('url', 'import.import.url');
		$form->addSelect('dataDownloader', 'import.import.dataDownloader', ['' => ''] + $this->dataDownloaders->getDownloadersOptionsForSelect());
		$form->addBool('allowUpload', 'import.import.allowUpload')->setDefaultValue(0);
		$form->addSelect('type', 'import.import.type', $this->importTypes->getSelectArr());

		$form->addSaveCancelControl();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->import instanceof Import) {
				$import = $this->import;

				$import->processingType = $values->processingType;

				$importType = $this->importTypes->getType($values->type);
				if ($importType instanceof ImportType) {
					$import->syncOpts = $importType->processSubmitSyncOpts($values->syncOpts);
					$import->data     = $importType->processSubmitData($values->data);
				}
			} else {
				$import = new Import($values->title, $values->type);
			}

			$import->title          = $values->title;
			$import->description    = $values->description;
			$import->isActive       = $values->isActive ? 1 : 0;
			$import->url            = $values->url;
			$import->dataDownloader = $values->dataDownloader;
			$import->allowUpload    = $values->allowUpload ? 1 : 0;
			$import->type           = $values->type;

			$flashMessage = $import->getId() ? 'import.import.edited' : 'import.import.added';

			$this->em->persist($import);
			$this->em->flush();
			$form->addCustomData('importId', $import->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setImport(int $id): void
	{
		$this->import = $this->importsService->get($id);

		if (!$this->import) {
			throw new BadRequestException;
		}

		$form   = $this['form'];
		$import = $this->import;

		$form->addSelect('processingType', 'import.import.processingType', $this->processingTypes->getSelectArr($import->type))
			->setDefaultValue('pairing');

		// TODO přidat do nastavení formuláře, teď se vše tahá z url, ale ta není vždy vyplněná.
		$importFileContent = $this->importsService->getFileContent($import);
		$importFileData    = $importFileContent ? $this->importsService->getFileData($importFileContent, $import) : [];

		$importType = $this->importTypes->getType($import->type);
		if ($importType instanceof ImportType) {
			$form->addComponent($importType->getFormSyncOpts((array) $importFileData['keys'], (array) $importFileData['values'], $import->getSyncOpts()), 'syncOpts');
			$form->addComponent($importType->getFormContainer((array) $importFileData['keys'], (array) $importFileData['values'], $import->data), 'data');
		}

		$this->onAnchor[] = function() use ($importFileData): void {
			$this->template->importFileData = ImportConfig::load('importForm.loadFileData') ? $importFileData : [];
		};

		$form->setDefaults([
			'title'       => $import->title,
			'description' => $import->description,
			'isActive'    => $import->isActive,
			'url'         => $import->url,
			'allowUpload' => $import->allowUpload,
		]);

		if ($import->dataDownloader && array_key_exists($import->dataDownloader, $form->getComponent('dataDownloader')->getItems())) {
			$form->getComponent('dataDownloader')->setDefaultValue($import->dataDownloader);
		}

		if ($import->processingType && array_key_exists($import->processingType, $form->getComponent('processingType')->getItems())) {
			$form->getComponent('processingType')->setDefaultValue($import->processingType);
		}

		if ($import->type && array_key_exists($import->type, $form->getComponent('type')->getItems())) {
			$form->getComponent('type')->setDefaultValue($import->type);
		}
	}

}
