<?php declare(strict_types = 1);

namespace Import\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Import\AdminModule\Model\Imports;
use Import\Model\Entities\Import;
use Import\Model\Entities\ImportHistory;
use Nette\Application\AbortException;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\Html;

class ImportsGrid extends BaseControl
{
	public function __construct(
		protected Imports $importsService,
	)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	/**
	 * @param int|string $id
	 */
	public function handleRun($id): void
	{
		bdump($id);
	}

	/**
	 * @param int|string $id
	 *
	 * @throws ForbiddenRequestException
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;

		if (!$presenter->getUser()->isAllowed('Import:Admin', 'create')) {
			throw new ForbiddenRequestException;
		}

		if ($this->importsService->remove($id)) {
			$presenter->flashMessageSuccess('default.deleted');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->importsService->getEr()->createQueryBuilder('i', 'i.id')
			->orderBy('i.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('importsGrid:gridSortableRow!');

		$lastRun = [];

		/** @var DoctrineDataSource $dataSource */
		$dataSource                 = $grid->getDataSource();
		$dataSource->onDataLoaded[] = function(array $items) use (&$lastRun): void {
			/** @var array<int, Import> $items */
			if ($items !== []) {
				$subQuery = $this->em->getRepository(ImportHistory::class)->createQueryBuilder('im2')
					->select('MAX(im2.id)')
					->where('im2.import IN (' . implode(',', array_keys($items)) . ')')
					->groupBy('im2.import')
					->getQuery();
				$expr     = $this->em->getExpressionBuilder();

				foreach ($this->em->getRepository(ImportHistory::class)->createQueryBuilder('im')
					         ->select('IDENTITY(im.import) as id, im.data, im.created')
					         ->where($expr->in('im.id', $subQuery->getDQL()))
					         ->getQuery()->getArrayResult() as $row) {
					$lastRun[$row['id']] = $row;
				}
			}
		};

		// Columns
		$grid->addColumnText('title', 'import.import.title')->setRenderer(function(Import $row) {
			$wrap = Html::el('div');

			if ($this->presenter->getUser()->isAllowed('Import:Admin', 'create')) {
				$wrap->addHtml(Html::el('a', ['href' => $this->presenter->link('Default:edit', $row->getId())])
					->setText($row->title));
			} else {
				$wrap->addHtml(Html::el('div', ['style' => 'font-weight: bold;'])->setText($row->title));
			}

			$wrap->addHtml(Html::el('div', ['style' => 'word-break: break-all; font-size: 12px;'])->setText($row->url));

			return $wrap;
		});
		$grid->addColumnStatus('isActive', 'default.isActive')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = $this->gridPublishChange(...);
		$grid->addColumnText('lastRun', 'import.import.lastRun')->setRenderer(function(Import $row) use (&$lastRun) {
			$result = Html::el();
			if (isset($lastRun[$row->getId()])) {
				$msg = '';
				foreach ($lastRun[$row->getId()]['data'] as $k => $v) {
					$msg .= $k . ' -> ' . $v . '<br>';
				}

				$result->addHtml(Html::el('div')->setText($lastRun[$row->getId()]['created']->format('Y-m-d H-i-s')))
					->addHtml(Html::el('div .text-muted', [
						'style' => 'font-size: 0.85em; word-break: break-all; max-width: 450px;',
					])->setHtml($msg));
			}

			return $result;
		});

		// Actions
		$grid->addAction('run', '', ':Import:Cron:Default:run')
			->setIcon('fas fa-play')
			->setBsType('primary')
			->setRenderCondition(fn() => $this->presenter->getUser()->isAllowed('Import:Admin', 'run'));
		$grid->addAction('delete', '', 'delete!')
			->setIcon('fas fa-times')
			->setBsType('danger')
			->setRenderCondition(fn() => $this->presenter->getUser()->isAllowed('Import:Admin', 'create'));

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->importsService->setActive((int) $id, (int) $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->importsService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

}
