<?php declare(strict_types = 1);

namespace Import\AdminModule\Model;

use Core\Model\Helpers\BaseEntityService;
use Core\Model\Helpers\Traits\TActive;
use Doctrine\ORM\Mapping\ClassMetadataInfo;
use Exception;
use Import\AdminModule\Model\Readers\CsvReader;
use Import\AdminModule\Model\Readers\JsonReader;
use Import\AdminModule\Model\Readers\XmlReader;
use Import\Model\Entities\Import;
use Nette\Utils\DateTime;
use Nette\Utils\FileSystem;
use Nette\Utils\Strings;

/**
 * @method Import|null getReference($id)
 * @method Import[] getAll()
 * @method Import|null get($id)
 */
class Imports extends BaseEntityService
{
	use TActive;

	protected         $entityClass       = Import::class;
	protected array   $fileContent       = [];
	protected array   $cRemoteLastUpdate = [];
	protected ?Import $selectedImport    = null;

	public function __construct(
		protected DataDownloaders $dataDownloaders,
	)
	{
	}

	/**
	 * @return mixed|null
	 */
	public function getFileContent(Import $import)
	{
		$this->selectedImport = $import;
		$result               = null;
		ini_set('memory_limit', '2G');

		if (!isset($this->fileContent[$import->getId()])) {
			if ($import->dataDownloader) {
				$downloader = $this->dataDownloaders->getDownloader($import->dataDownloader);
				if (!$downloader) {
					return null;
				}
				$file   = $downloader->getFile($import);

				if (!$file) {
					return null;
				}

				$body   = $file ? @file_get_contents($file) : '';
				$result = [
					'contentType' => \str_starts_with((string) $body, '<') ? 'text/xml' : mime_content_type($file),
					'extension'   => pathinfo($file, PATHINFO_EXTENSION),
					'body'        => $body,
				];
			} else if ($import->url) {
				$file   = $this->downloadFile($import->url);
				$body   = $file !== '' && $file !== '0' ? @file_get_contents($file) : '';
				$result = [
					'contentType' => \str_starts_with((string) $body, '<') ? 'text/xml' : mime_content_type($file),
					'extension'   => pathinfo($import->url, PATHINFO_EXTENSION),
					'body'        => $body,
				];
			}
		}

		if ($result) {
			$this->fileContent[$import->getId()] = $result;
		}

		return $this->fileContent[$import->getId()] ?? null;
	}

	public function getFileData(array $fileContent, Import $import): array
	{
		$importFileData = ['data' => [], 'keys' => [], 'values' => []];
		$reader         = null;

		switch ($fileContent['contentType']) {
			case 'text/xml':
				$reader = new XmlReader($fileContent['body']);
				if ($import->getSyncOpt('xPath')) {
					$reader->setXpath($import->getSyncOpt('xPath'));
				}
				break;
			case 'text/plain':
			case 'application/json':
				$reader = new JsonReader($fileContent['body']);
				break;
		}

		if (!$reader && $fileContent['extension'] === 'csv') {
			$reader = new CsvReader($fileContent['body']);
		}

		if ($reader) {
			$importFileData['data']   = $reader->getData();
			$importFileData['keys']   = $reader->getKeys();
			$importFileData['values'] = $reader->getValues();
		}

		return $importFileData;
	}

	public function remoteLastUpdate(string $url): int
	{
		$import = $this->selectedImport;

		$extension = pathinfo($url, PATHINFO_EXTENSION);
		if ($extension == 'csv') {
			$this->cRemoteLastUpdate[$url] = time();
		}

		if (!isset($this->cRemoteLastUpdate[$url])) {
			$curl = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_NOBODY, true);
			curl_setopt($curl, CURLOPT_HEADER, true);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_FILETIME, true);
			curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'GET');

			$username = $import->getSyncOpt('username');
			$password = $import->getSyncOpt('password');
			if ($username && $password) {
				curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
				curl_setopt($curl, CURLOPT_USERPWD, $username . ':' . $password);
			}

			$result = curl_exec($curl);
			$info   = curl_getinfo($curl);

			curl_close($curl);

			$this->cRemoteLastUpdate[$url] = $info['filetime'];
		}

		return $this->cRemoteLastUpdate[$url];
	}

	public function needDownload(string $url): bool
	{
		$tmpFile = $this->getTmpFile($url, $this->remoteLastUpdate($url));
		if (file_exists($tmpFile)) {
			$lastUpdate = filemtime($tmpFile);
			$lastUpdate = (DateTime::from((string) $lastUpdate))->format('Y-m-d H:i:s');
			$compare    = (new DateTime)->modify('-1 minute')->format('Y-m-d H:i:s');

			return $compare > $lastUpdate;
		}

		return true;
	}

	public function needUpdate(array $import): bool
	{
		return true;
	}

	public function addHistory(int $id, array $data): void
	{
		$this->em->getConnection()->executeQuery("INSERT INTO import__import (import_id, status, created, `data`) VALUES (?, ?, ?, ?)", [$id, 'ok', new DateTime, serialize($data)]);
	}

	public function updateLastRun(int $id, string $url): void
	{
		$this->em->getConnection()->executeQuery("UPDATE import__import SET last_run = ? WHERE id = ?", [DateTime::from($this->remoteLastUpdate($url)), $id]);
	}

	protected function getTmpFile(string $url, int $modified): string { return TMP_DIR . '/imports/' . Strings::webalize($url) . '__m' . $modified; }

	/**
	 * Stáhne se soubor podle importu podle data modifikace
	 */
	protected function downloadFile(string $url): string
	{
		$tmpFile = $this->getTmpFile($url, $this->remoteLastUpdate($url));
		if ($this->needDownload($url)) {
			foreach (glob(TMP_DIR . '/imports/' . Strings::webalize($url) . '*') as $v) {
				unlink($v);
			}

			$import = $this->selectedImport;
			$curl   = curl_init();
			curl_setopt($curl, CURLOPT_URL, $url);
			curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
			curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'GET');
			curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);

			$username = $import->getSyncOpt('username');
			$password = $import->getSyncOpt('password');
			if ($username && $password) {
				curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
				curl_setopt($curl, CURLOPT_USERPWD, $username . ':' . $password);
			}


			$result = curl_exec($curl);
			curl_close($curl);

			if ($result) {
				FileSystem::write($tmpFile, $result);
			}
		}

		return $tmpFile;
	}

	/**
	 * @param int $id
	 * @param int $position
	 *
	 * @throws Exception
	 */
	public function setPosition($id, $position): bool
	{
		if ($item = $this->get($id)) {
			$item->setPosition($position);
			$this->em->persist($item);
			$this->em->flush();

			return true;
		}

		return false;
	}

	public function getByGroup(int $groupId): array
	{
		return $this->getEr()->createQueryBuilder('i')
			->where('i.group = :group')
			->setParameter('group', $groupId)
			->getQuery()->getArrayResult();
	}

	public function getAsArray(int $id): ?array
	{
		return $this->getEr()->createQueryBuilder('i')
			->where('i.id = :id')
			->setParameter('id', $id)
			->setMaxResults(1)
			->getQuery()->getArrayResult()[0] ?? null;
	}

	/**
	 * Clean for memory usage
	 */
	public function clean(): void
	{
		$this->cRemoteLastUpdate = [];
		$this->fileContent       = [];
	}
}
