<?php declare(strict_types = 1);

namespace Import\Console;

use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Doctrine;
use Exception;
use Import\AdminModule\Model\Imports;
use Import\AdminModule\Model\ProcessingTypes;
use Nette\Utils\Json;
use Override;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Tracy\Debugger;

#[AsCommand(name: 'imports:run')]
class Run extends Command
{
	public function __construct(
		protected Imports         $imports,
		protected ProcessingTypes $processingTypes,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('imports:run')
			->addOption('id', null, InputOption::VALUE_REQUIRED)
			->setDescription('Run import');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('<fg=white;bg=#007bff>IMPORT - Run</>');
		try {
			ini_set('memory_limit', '2G');
			$import = $this->imports->getAsArray((int) $input->getOption('id'));

			if ($import && $import['isActive']) {

				$processingType = $this->processingTypes->getType($import['type'], $import['processingType']);
				if ($processingType && $import['id']) {
					$importEntity = $this->imports->get($import['id']);

					if (!$importEntity) {
						throw new Exception('Import entity not found: ' . $import['id']);
					}

					$fileContent = $this->imports->getFileContent($importEntity);
					$fileData    = $fileContent ? $this->imports->getFileData($fileContent, $importEntity) : [];

					$responseData = $processingType->run($importEntity, $fileData);
					$this->imports->updateLastRun($importEntity->getId(), $importEntity->url);
					$this->imports->addHistory($importEntity->getId(), $responseData);

					$output->writeLn('<info>[OK] - Import done</info>');

					return Command::SUCCESS;
				}

				$output->writeln('<error>[ERROR] - Processing type not found</error>');
			} else {
				$output->writeln('<error>[ERROR] - Not found or is not active</error>');
			}
		} catch (Exception $e) {
			Debugger::log($e, 'importRun');
			LogNotifier::toDevelopers('Import - Run <br>' . Json::encode($input->getOptions()) . '<br>' . $e->getMessage(), 'Import run error');

			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return Command::FAILURE;
	}
}
