<?php declare(strict_types = 1);

namespace Import\Console;

use Doctrine;
use Import\AdminModule\Model\Imports;
use Import\AdminModule\Model\ProcessingType\ProcessingType;
use Import\AdminModule\Model\ProcessingTypes;
use Override;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand(name: 'imports:run-group')]
class RunGroup extends Command
{
	public function __construct(
		protected Imports         $imports,
		protected ProcessingTypes $processingTypes,
	)
	{
		parent::__construct();
	}

	#[Override]
	protected function configure(): void
	{
		$this->setName('imports:run-group')
			->addOption('id', null, InputOption::VALUE_REQUIRED)
			->setDescription('Run group import');
	}

	#[Override]
	protected function execute(
		InputInterface  $input,
		OutputInterface $output,
	): int
	{
		$output->writeln('<fg=white;bg=#007bff>IMPORT - Run group</>');
		try {
			ini_set('memory_limit', '2G');
			$imports = $this->imports->getByGroup((int) $input->getOption('id'));

			foreach ($imports as $import) {
				if ($import && $import['isActive']) {

					$processingType = $this->processingTypes->getType($import['type'], $import['processingType']);
					if ($processingType instanceof ProcessingType) {
						$import       = $this->imports->getReference($import['id']);
						$fileContent  = $this->imports->getFileContent($import);
						$fileData     = $fileContent ? $this->imports->getFileData($fileContent, $import) : [];
						$responseData = $processingType->run($import, $fileData);
						$this->imports->updateLastRun($import->getId(), $import->url);
						$this->imports->addHistory($import->getId(), $responseData);

						$output->writeLn('<info>[OK] - Import done</info>');
					} else {
						$output->writeln('<error>[ERROR] - Processing type not found</error>');
					}
				} else {
					$output->writeln('<error>[ERROR] - Not found or is not active</error>');
				}
			}
		} catch (\Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return 1;
	}

}
