<?php declare(strict_types = 1);

namespace Import\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Import\AdminModule\Model\ImportGroups;
use Nette\Application\AbortException;
use Nette\Application\ForbiddenRequestException;

class ImportGroupsGrid extends BaseControl
{
	protected ImportGroups $importGroupsService;

	public function __construct(
		ImportGroups $importGroup
	)
	{
		$this->importGroupsService = $importGroup;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	/**
	 * @param string|int $id
	 */
	public function handleRun($id): void
	{
		bdump($id);
	}

	/**
	 * @param string|int $id
	 *
	 * @throws AbortException
	 * @throws ForbiddenRequestException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;

		if (!$presenter->getUser()->isAllowed('Import:Admin', 'create')) {
			throw new ForbiddenRequestException;
		}

		if ($this->importGroupsService->remove($id)) {
			$presenter->flashMessageSuccess('default.deleted');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->importGroupsService->getEr()->createQueryBuilder('g');
		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnLink('title', 'import.import.title', 'Default:editGroup');
		$grid->addColumnStatus('isActive', 'default.isActive')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];

		// Actions
		$grid->addAction('run', '', ':Import:Cron:Default:runGroup')->setIcon('fas fa-play')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('fas fa-times')->setBsType('danger')
			->setRenderCondition(function() { return $this->presenter->getUser()->isAllowed('Import:Admin', 'create'); });

		// Prototype
		$grid->getColumn('isActive')->getElementPrototype('td')->class[] = 'w1nw';

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->importGroupsService->setActive($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

}
