<?php declare(strict_types = 1);

namespace Import\AdminModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\Strings;
use Core\Model\Http\Session;
use EshopCatalog\Model\Entities\Product;
use Exception;
use Import\Model\Entities\Import;
use Nette\Caching\Cache;
use Nette\Caching\Storage;
use Nette\Http\SessionSection;

class PriceCheck
{
	public const CACHE_NAMESPACE   = 'importPriceCheck';
	public const SESSION_NAMESPACE = 'Import/PriceCheck';

	protected EntityManagerDecorator $em;
	protected Cache                  $cache;
	protected Imports                $imports;
	protected Session                $session;

	public function __construct(
		EntityManagerDecorator $em,
		Storage                $storage,
		Imports                $imports,
		Session                $session
	)
	{
		$this->em      = $em;
		$this->cache   = new Cache($storage, self::CACHE_NAMESPACE);
		$this->imports = $imports;
		$this->session = $session;
	}

	public function getSession(): SessionSection { return $this->session->getSection(self::SESSION_NAMESPACE); }

	public function getEnabledSuppliers(): array
	{
		return (array) $this->getSession()->enabledSuppliers ?: [];
	}

	/**
	 * @return Import[]
	 */
	public function getImports(): array
	{
		$arr = [];

		foreach ($this->imports->getEr()->createQueryBuilder('i')
			         ->where('i.data LIKE :data')
			         ->setParameter('data', "%purchasePrice%")
			         ->getQuery()->getResult() as $row) {
			/** @var Import $row */
			if ($row->data['purchasePrice']['key'] && ($row->data['vatRate']['key'] || $row->data['vatRate']['fixedValue'])) {
				$arr[$row->getSyncOpt('supplier')] = $row;
			}
		}

		return $arr;
	}

	public function loadPrices(): void
	{
		$arr              = [];
		$enabledSuppliers = $this->getEnabledSuppliers();

		foreach ($this->getImports() as $import) {
			if (!in_array($import->getSyncOpt('supplier'), $enabledSuppliers)
				|| !isset($import->data['purchasePrice']['key'])
				|| empty($import->data['purchasePrice']['key'])) {
				continue;
			}

			$fileContent  = $this->imports->getFileContent($import);
			$data         = $fileContent ? $this->imports->getFileData($fileContent, $import) : [];
			$supplierName = $import->getSyncOpt('supplier');
			foreach ($data['data'] as $row) {
				$supplierCode  = $row[$import->data['supplierCode']['key']];
				$key           = $supplierName . '_' . $supplierCode;
				$purchasePrice = $row[$import->data['purchasePrice']['key']];
				$vat           = $row[$import->data['vatRate']['key']] ?? $import->data['vatRate']['fixedValue'];

				$arr[$key] = [
					'supplierCode'  => $supplierCode,
					'purchasePrice' => $purchasePrice,
					'vat'           => $vat,
				];
			}
		}

		$this->cache->save('purchasePrice', $arr);
	}

	public function getPrices(): array { return $this->cache->load('purchasePrice') ?: []; }

	public function changePrice(int $productId, float $price, ?float $retailPrice): bool
	{
		try {
			/** @var Product $product */
			$product = $this->em->getReference(Product::class, $productId);

			$tmp            = Strings::formatEntityDecimal($price);
			$product->price = $tmp ? (float) $tmp : null;

			if ($retailPrice) {
				$tmp = Strings::formatEntityDecimal($retailPrice);

				$product->retailPrice = $tmp ? (float) $tmp : null;
			}

			$this->em->persist($product);
			$this->em->flush();

			return true;
		} catch (Exception $e) {
		}

		return false;
	}
}
