<?php declare(strict_types = 1);

namespace Import\AdminModule\Model\Readers;

class CsvReader implements IReader
{
	protected ?string $fileContent = null;
	protected ?array $cachedRawData = null;

	public function __construct(string $fileContent)
	{
		$this->fileContent = $fileContent;
	}

	public function getKeys(): array
	{
		return $this->getRawData()['header'];
	}

	public function getValues(): array
	{
		$result = [];
		foreach ($this->getRawData()['body'] as $row) {
			foreach ($row as $rk => $rv) {
				if (is_string($rv) || is_numeric($rv)) {
					$result[$rk][] = $rv;
				}
			}
		}

		foreach ($result as $k => $v) {
			$result[$k] = array_chunk(array_unique($v), 100)[0];
		}

		return $result;
	}

	protected function getRawData(): ?array
	{
		if ($this->cachedRawData === null) {
			$header = [];
			$body   = [];
			foreach (explode("\r\n", $this->fileContent) as $rowK => $rowString) {
				$vals = explode(';', $rowString);

				if ($rowK === 0) {
					$header = $vals;
				}
				else {
					$body[] = $vals;
				}
			}

			$uniqueHeader = array_unique($header);
			if (count($uniqueHeader) <= 2 && (in_array('', $uniqueHeader) || in_array('0', $uniqueHeader))) {
				$tmp    = array_values($header);
				$header = [];
				foreach ($tmp as $k => $v) {
					$header[$k] = $k;
				}
			}
			$this->cachedRawData = [
				'header' => $header,
				'body'   => $body,
			];
			$this->fileContent = null;
		}

		return $this->cachedRawData;
	}

	public function getData(): array
	{
		return $this->getRawData()['body'];
	}

}
