<?php declare(strict_types = 1);

namespace Import\AdminModule\Model\Readers;

use Core\Model\Helpers\Arrays;
use Nette\Utils\FileSystem;
use Nette\Utils\Json;

class JsonReader implements IReader
{
	protected ?string $fileContent = null;
	protected ?array $cachedRawData = null;

	public function __construct(string $fileContent)
	{
		$this->fileContent = $fileContent;
	}

	public function getKeys(): array
	{
		$keys = [];

		foreach ($this->getRawData()['header'] as $row) {
			if (!is_array($row)) {
				$keys[$row] = $row;
			} else {
				foreach (array_keys($row) as $k) {
					$keys[$k] = $k;
				}
			}
		}

		return $keys;
	}

	public function getValues(): array
	{
		$result = [];
		foreach ($this->getRawData()['body'] as $row) {
			foreach ($row as $rk => $rv) {
				if (is_string($rv) || is_numeric($rv)) {
					$result[$rk][] = $rv;
				}
			}
		}

		foreach ($result as $k => $v) {
			$result[$k] = array_chunk(array_unique($v), 100)[0];
		}

		return $result;
	}

	protected function getRawData(): ?array
	{
		if ($this->cachedRawData === null) {
			$header = [];
			$body   = [];
			foreach ($this->decodeJson($this->fileContent) as $k => $v) {
				if (!is_array($v)) {
					continue;
				}
				$header   = array_merge($header, array_keys($v));
				$body[$k] = $v;
			}

			$this->cachedRawData = [
				'header' => $header,
				'body'   => $body,
			];
			$this->fileContent   = null;
		}

		return $this->cachedRawData;
	}

	public function getData(): array
	{
		return $this->getRawData()['body'];
	}

	protected function decodeJson(string $content): array {
		if (Arrays::isJson($content)) {
			return Json::decode($content, Json::FORCE_ARRAY);
		}

		$data = [];
		foreach (explode("\r", $content) as $line) {
			if ($line) {
				$data[] = Json::decode($line, Json::FORCE_ARRAY);
			}
		}
		return $data;
	}
}
