<?php declare(strict_types = 1);

namespace Import\Console;

use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Import\AdminModule\Model\Imports;
use Import\AdminModule\Model\ProcessingTypes;
use Nette\Utils\Json;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Tracy\Debugger;

class Run extends AbstractCommand
{
	/** @var Imports|null @inject */
	public ?Imports $imports = null;

	/** @var ProcessingTypes|null @inject */
	public ?ProcessingTypes $processingTypes = null;

	protected function configure(): void
	{
		$this->setName('imports:run')
			->addOption('id', null, InputOption::VALUE_REQUIRED)
			->setDescription('Run import');
	}

	protected function executeSchemaCommand(InputInterface $input, OutputInterface $output, Doctrine\ORM\Tools\SchemaTool $schemaTool, array $metadatas, SymfonyStyle $ui): int
	{
		$output->writeln('<fg=white;bg=#007bff>IMPORT - Run</>');
		try {
			ini_set('memory_limit', '2G');
			$import = $this->imports->getAsArray((int) $input->getOption('id'));

			if ($import && $import['isActive']) {

				$processingType = $this->processingTypes->getType($import['type'], $import['processingType']);
				if ($processingType && $import['id']) {
					$importEntity = $this->imports->get($import['id']);

					if (!$importEntity) {
						throw new \Exception('Import entity not found: ' . $import['id']);
					}

					$fileContent = $this->imports->getFileContent($importEntity);
					$fileData    = $fileContent ? $this->imports->getFileData($fileContent, $importEntity) : [];

					$responseData = $processingType->run($importEntity, $fileData);
					$this->imports->updateLastRun($importEntity->getId(), $importEntity->url);
					$this->imports->addHistory($importEntity->getId(), $responseData);

					$output->writeLn('<info>[OK] - Import done</info>');

					return Command::SUCCESS;
				}

				$output->writeln('<error>[ERROR] - Processing type not found</error>');
			} else {
				$output->writeln('<error>[ERROR] - Not found or is not active</error>');
			}
		} catch (\Exception $e) {
			Debugger::log($e, 'importRun');
			LogNotifier::toDevelopers('Import - Run <br>' . Json::encode($input->getOptions()) . '<br>' . $e->getMessage(), 'Import run error');

			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return Command::FAILURE;
	}
}
