<?php declare(strict_types = 1);

namespace Import\CronModule\Presenters;

use Exception;
use Import\AdminModule\Model\ImportGroups;
use Import\AdminModule\Model\Imports;
use Import\AdminModule\Model\ProcessingType\IProcessingType;
use Import\AdminModule\Model\ProcessingTypes;
use Core\Model\Entities\EntityManagerDecorator;

class DefaultPresenter extends BasePresenter
{
	/** @var Imports @inject */
	public Imports $importsService;

	/** @var ImportGroups @inject */
	public ImportGroups $importGroupsService;

	/** @var ProcessingTypes @inject */
	public ProcessingTypes $processingTypes;

	/** @var EntityManagerDecorator @inject */
	public EntityManagerDecorator $em;

	public function actionRun(int $id): void
	{
		$import       = $this->importsService->getAsArray($id);
		$responseData = $this->runImport($import);

		$this->sendJson($responseData);
	}

	public function actionRunGroup(int $id): void
	{
		$imports        = $this->importsService->getByGroup($id);
		$resultResponse = ['status' => 'error', 'message' => '', 'imports' => []];

		if (!$imports) {
			$resultResponse['message'] = "Import id '$id' not found";
			$this->sendJson($resultResponse);
		}

		foreach ($imports as $import) {
			if ($import['isActive']) {
				$resultResponse['imports'][$import['title']] = $this->runImport($import);
			}
		}

		if (!empty($resultResponse['imports'])) {
			$resultResponse['status'] = 'ok';
		}

		foreach ($resultResponse['imports'] as $data) {
			if ($data['status'] == 'error') {
				$resultResponse['status'] = 'error';
				break;
			}
		}

		$this->sendJson($resultResponse);
	}

	protected function runImport(array $import): array
	{
		$importId     = $import['id'];
		$responseData = ['status' => 'error', 'message' => ''];
		if (!$import || !$import['type'] || !$import['processingType'] || !$import['isActive']) {
			$responseData['message'] = "Import id '$importId' not found";

			return $responseData;
		}

		$processingType = $this->processingTypes->getType($import['type'], $import['processingType']);
		if (!$processingType) {
			$responseData['message'] = "Import (id: '$importId') processing type error";

			return $responseData;
		}

		try {
			if ($this->importsService->needUpdate($import)) {
				$import       = $this->importsService->getReference($import['id']);
				$fileContent  = $this->importsService->getFileContent($import);
				$fileData     = $fileContent ? $this->importsService->getFileData($fileContent, $import) : [];
				$responseData = $processingType->run($import, $fileData);
				$this->importsService->updateLastRun($import->getId(), $import->url);
				$this->importsService->addHistory($import->getId(), $responseData);
			} else {
				$this->importsService->addHistory($import['id'], ['message' => 'No new data']);
			}

			$responseData['status'] = 'ok';
		} catch (Exception $e) {
			$responseData['message'] = $e->getMessage();
		}

		$this->importsService->clean();

		return $responseData;
	}

}
