<?php declare(strict_types = 1);

namespace Import\Model\Entities;

use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Core\Model\Entities\TId;
use Nette\Utils\ArrayHash;
use Nette\Utils\DateTime;
use Nette\Utils\Json;

/**
 * @ORM\Table(name="import__import")
 * @ORM\Entity
 */
class Import
{
	use TId;

	/**
	 * @ORM\Column(type="string", length=255, nullable=false)
	 */
	public string $title;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $description = null;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 1})
	 */
	public int $isActive;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $url = null;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $dataDownloader = null;

	/**
	 * @ORM\Column(type="smallint", nullable=false, options={"default": 0})
	 */
	public int $allowUpload;

	/**
	 * @ORM\Column(type="string", length=255, nullable=false)
	 */
	public string $type;

	/**
	 * @ORM\Column(type="string", length=255, nullable=true)
	 */
	public ?string $processingType = null;

	/**
	 * @var array|ArrayHash
	 * @ORM\Column(type="array", nullable=true)
	 */
	public $data = null;

	/**
	 * @var DateTime|null
	 * @ORM\Column(type="datetime", nullable=true)
	 */
	public $lastRun;

	/**
	 * @var array|ArrayHash|null
	 * @ORM\Column(type="array", nullable=true)
	 */
	public $syncOpts = null;

	/**
	 * @var Collection<ImportHistory>
	 * @ORM\OneToMany(targetEntity="ImportHistory", mappedBy="import", orphanRemoval=true)
	 * @ORM\OrderBy({"created"="DESC"})
	 */
	private Collection $history;

	/**
	 * @ORM\ManyToOne(targetEntity="ImportGroup", inversedBy="imports", cascade={"persist"})
	 * @ORM\JoinColumn(name="group_id", referencedColumnName="id", onDelete="SET NULL")
	 */
	protected ?ImportGroup $group = null;

	/**
	 * @Gedmo\SortablePosition
	 * @ORM\Column(type="integer")
	 */
	private int $position;

	public function __construct(string $title, string $type)
	{
		$this->title       = $title;
		$this->isActive    = 1;
		$this->allowUpload = 0;
		$this->type        = $type;
		$this->data        = [];
		$this->syncOpts    = [];
		$this->history     = new ArrayCollection;
		$this->setPosition(-1);
	}

	/*********************
	 * == SyncOpts
	 */

	/**
	 * @return array|ArrayHash|null
	 */
	public function getSyncOpts() { return $this->syncOpts; }

	/**
	 * @param int|string $key
	 *
	 * @return int|string|float|array|null
	 */
	public function getSyncOpt($key) { return $this->syncOpts[$key] ?? null; }

	public function getSyncOptString(string $key): string
	{
		$v = $this->getSyncOpt($key);

		if (is_array($v)) {
			return Json::encode($v);
		}

		return (string) $v;
	}

	/**
	 * @param int|string                  $key
	 * @param int|string|float|array|null $value
	 */
	public function setSyncOpt($key, $value = null): void
	{
		if ($value == null) {
			unset($this->syncOpts[$key]);
		} else {
			$this->syncOpts[$key] = $value;
		}
	}

	/*****************
	 * == History
	 */

	/**
	 * @return Collection<ImportHistory>
	 */
	public function getHistory(): Collection { return $this->history; }

	/******
	 * === Position
	 */

	/**
	 * @param int|string $position
	 */
	public function setPosition($position): void
	{
		$this->position = (int) $position;
	}

	public function getPosition(): int
	{
		return $this->position;
	}

	/******
	 * === Group
	 */

	public function setGroup(?ImportGroup $group): Import
	{
		$this->group = $group;

		return $this;
	}

	public function getGroup(): ?ImportGroup { return $this->group; }
}
