<?php declare(strict_types = 1);

namespace Import\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\DataGrid\DataSource\DoctrineDataSource;
use Import\AdminModule\Model\Imports;
use Import\Model\Entities\Import;
use Import\Model\Entities\ImportHistory;
use Nette\Application\AbortException;
use Nette\Application\ForbiddenRequestException;
use Nette\Utils\Html;

class ImportsGrid extends BaseControl
{
	public function __construct(protected Imports $importsService)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */
	public function handleRun(int|string $id): void
	{
		bdump($id);
	}

	/**
	 *
	 * @throws ForbiddenRequestException
	 * @throws AbortException
	 */
	public function handleDelete(int|string $id): void
	{
		$presenter = $this->presenter;

		if (!$presenter->getUser()->isAllowed('Import:Admin', 'create')) {
			throw new ForbiddenRequestException;
		}

		if ($this->importsService->remove($id)) {
			$presenter->flashMessageSuccess('default.deleted');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * =================  Components
	 */

	public function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->importsService->getEr()->createQueryBuilder('i', 'i.id')
			->orderBy('i.position');
		$grid->setDataSource($qb);

		$grid->setSortable();
		$grid->setSortableHandler('importsGrid:gridSortableRow!');

		$lastRun = [];

		/** @var DoctrineDataSource $dataSource */
		$dataSource                 = $grid->getDataSource();
		$dataSource->onDataLoaded[] = function(array $items) use (&$lastRun) {
			/** @var array<int, Import> $items */
			if (!empty($items)) {
				$subQuery = $this->em->getRepository(ImportHistory::class)->createQueryBuilder('im2')
					->select('MAX(im2.id)')
					->where('im2.import IN (' . implode(',', array_keys($items)) . ')')
					->groupBy('im2.import')
					->getQuery();
				$expr     = $this->em->getExpressionBuilder();

				foreach ($this->em->getRepository(ImportHistory::class)->createQueryBuilder('im')
					         ->select('IDENTITY(im.import) as id, im.data, im.created')
					         ->where($expr->in('im.id', $subQuery->getDQL()))
					         ->getQuery()->getArrayResult() as $row) {
					$lastRun[$row['id']] = $row;
				}
			}
		};

		// Columns
		$grid->addColumnText('title', 'import.import.title')->setRenderer(function(Import $row) {
			if ($this->presenter->getUser()->isAllowed('Import:Admin', 'create')) {
				return Html::el('a', ['href' => $this->presenter->link('Default:edit', $row->getId())])
					->setText($row->title);
			}

			return $row->title;
		});
		$grid->addColumnStatus('isActive', 'default.isActive')
			->setAlign('center')
			->addOption(1, 'default.publish')
			->setIcon('check')
			->setClass('btn-success')
			->setShowTitle(false)
			->endOption()
			->addOption(0, 'default.unPublish')
			->setIcon('times')
			->setClass('btn-danger')
			->setShowTitle(false)
			->endOption()
			->onChange[] = $this->gridPublishChange(...);
		$grid->addColumnText('type', 'import.import.type');
		$grid->addColumnText('lastRun', 'import.import.lastRun')->setRenderer(function(Import $row) use (&$lastRun) {
			$result = Html::el();
			if (isset($lastRun[$row->getId()])) {
				$msg = '';
				foreach ($lastRun[$row->getId()]['data'] as $k => $v) {
					$msg .= $k . ' -> ' . $v . '<br>';
				}

				$result->addHtml(Html::el('div')->setText($lastRun[$row->getId()]['created']->format('Y-m-d H-i-s')))
					->addHtml(Html::el('dic .text-muted')->setHtml($msg));
			}

			return $result;
		});

		// Actions
		$grid->addAction('run', '', ':Import:Cron:Default:run')->setIcon('fas fa-play')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setIcon('fas fa-times')->setBsType('danger')
			->setRenderCondition(fn() => $this->presenter->getUser()->isAllowed('Import:Admin', 'create'));

		return $grid;
	}

	/*******************************************************************************************************************
	 * =================  Grid function
	 */
	/**
	 * @param int $id
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, int|bool $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->importsService->setActive((int) $id, (int) $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	public function handleGridSortableRow(): void
	{
		$presenter = $this->presenter;
		$request   = $presenter->getHttpRequest();
		$id        = $request->getPost('id');
		$position  = $request->getPost('position');

		if ($id != null && $position != null && $this->importsService->setPosition($id, $position)) {
			$presenter->flashMessageSuccess('default.positionChanged');
		} else {
			$presenter->flashMessageDanger('default.positionChangeFailed');
		}

		$this['grid']->reload();
		$presenter->redrawControl('flashes');
	}

}
