<?php declare(strict_types = 1);

namespace Import\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use Core\Model\UI\Form\BaseForm;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\Product;
use EshopCatalog\Model\Entities\ProductSupplier;
use Import\AdminModule\Model\PriceCheck;
use Nette\Utils\ArrayHash;
use Nette\Utils\Html;

class PriceCheckGrid extends BaseControl
{
	public function __construct(protected PriceCheck $priceCheck)
	{
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	public function reload(): void
	{
		$this['grid']->setDataSource($this->getData());
		$this['grid']->reload();
	}

	/*******************************************************************************************************************
	 * ========= Handle
	 */

	public function handleChangePrice(): void
	{
		$presenter   = $this->presenter;
		$id          = $this->getPost('id');
		$price       = $this->getPost('price');
		$retailPrice = $this->getPost('retailPrice');

		if ($retailPrice) {
			$retailPrice = (float) $retailPrice;
		}

		if (!$price) {
			$presenter->flashMessageDanger('import.priceCheck.fillNewPrice');
		} else if ($id && $this->priceCheck->changePrice((int) $id, (float) $price, $retailPrice)) {
			$presenter->flashMessageSuccess('import.priceCheck.priceChanged');
			$this['grid']->setDataSource($this->getData([$id]));
			$this['grid']->redrawItem((int) $id);
		} else {
			$presenter->flashMessageDanger('default.error');
		}

		$presenter->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ========= Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$grid->setDataSource($this->getData());

		// Columns
		$grid->addColumnLink(
			'link',
			'import.priceCheck.productName',
			':EshopCatalog:Admin:Products:edit',
			'name',
			['id' => 'id'],
		);
		$grid->addColumnNumber('localQuantity', 'import.priceCheck.localQuantity')->setRenderer(function(array $row) {
			if ($row['localQuantity'] > 0) {
				$el = Html::el('span', ['class' => 'text-success']);
			} else {
				$el = Html::el('b', ['class' => 'text-danger']);
			}
			$el->setText($row['localQuantity']);

			return $el;
		});
		$grid->addColumnNumber('price', 'import.priceCheck.productPrice')->setRenderer(fn(array $row) => Html::el('b')
			->setText(number_format($row['price'], 2, ',', ' ')))
			->setAlign('right');
		$grid->addColumnText('supplier', 'import.priceCheck.supplierPurchasePrice')->setRenderer(function(array $row) {
			$wrap = Html::el();
			foreach ($row['codes'] as $code) {
				$price = number_format((float) $code['purchasePrice'], 2, ',', ' ');
				$wrap->addHtml(
					Html::el('div')
						->setHtml($code['supplier'] . ' - ' . $code['code'] . ' - <b>' . $price . '</b>'),
				);
			}

			return $wrap;
		})->setAlign('right');
		$grid->addColumnText('diff', 'import.priceCheck.diff')->setRenderer(function(array $row) {

			$wrap = Html::el();
			foreach ($row['codes'] as $code) {
				$color = 'success';
				$class = 'text-bold';

				if ($code['diff'] < 15) {
					$color = 'danger';
				}
				$class .= ' text-' . $color;

				$wrap->addHtml(
					Html::el('div', ['class' => $class])
						->setText(number_format($code['diff'], 2, ',', ' ') . '%'),
				);
			}

			return $wrap;
		})->setAlign('right');
		$grid->addColumnText('change', 'import.priceCheck.priceChange')->setRenderer(function(array $row) {
			$wrap   = Html::el('div', ['class' => 'd-flex']);
			$input  = Html::el('input', [
				'placeholder' => $row['price'],
				'name'        => 'price',
				'style'       => 'width: 100px; margin-right: 2px;',
			]);
			$input2 = Html::el('input', [
				'placeholder' => $row['retailPrice'],
				'name'        => 'retailPrice',
				'style'       => 'width: 100px; margin-right: 2px;',
			]);
			$btn    = Html::el('a', [
				'href'    => 'javascript:;',
				'data-id' => $row['id'],
				'class'   => 'btn btn-success change-price',
			])->setText($this->t('default.save'));

			$wrap->addHtml($input)
				->addHtml($input2)
				->addHtml($btn);

			return $wrap;
		});

		// Prototype
		$grid->getColumn('localQuantity')->getElementPrototype('th')->class[] = 'w1nw';
		$grid->getColumn('price')->getElementPrototype('th')->class[]         = 'w1nw';
		$grid->getColumn('supplier')->getElementPrototype('th')->class[]      = 'w1nw';
		$grid->getColumn('supplier')->getElementPrototype('td')->class[]      = 'w1nw';
		$grid->getColumn('diff')->getElementPrototype('td')->class[]          = 'w1nw';
		$grid->getColumn('change')->getElementPrototype('td')->class[]        = 'w1nw';

		return $grid;
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setShowLangSwitcher(false);

		$defaults  = $this->priceCheck->getEnabledSuppliers();
		$suppliers = [];
		foreach ($this->priceCheck->getImports() as $import) {
			$suppliers[$import->getSyncOpt('supplier')] = $import->getSyncOpt('supplier');
		}

		asort($suppliers);

		if (!$defaults) {
			$defaults = array_keys($suppliers);
		}

		$form->addCheckboxList('suppliers', $this->t('import.priceCheck.suppliers'), $suppliers)
			->setTranslator(null)
			->setDefaultValue($defaults);

		$form->addSubmit('save', 'default.save');

		$form->onSuccess[] = function(BaseForm $form, ArrayHash $values) {
			$this->priceCheck->getSession()->enabledSuppliers = (array) $values->suppliers;

			$this->presenter->flashMessageSuccess('default.saved');
			$this->presenter->redrawControl('flashes');
			$this->reload();
		};

		return $form;
	}

	protected function getData(array $ids = []): array
	{
		$ids              = array_splice($ids, 0, 20);
		$products         = [];
		$enabledSuppliers = $this->priceCheck->getEnabledSuppliers();

		$qb = $this->em->getRepository(Product::class)->createQueryBuilder('p')
			->select('p.id, p.retailPrice, p.price, pt.name, p.quantity')
			->innerJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
			->where('p.isPublished = 1')
			->andWhere('p.price IS NOT NULL')
			->andWhere('p.price > 0')
			->setParameters([
				'lang' => $this->translator->getLocale(),
			])
			->groupBy('p.id');

		if (!empty($ids)) {
			$qb->andWhere('p.id IN (:ids)')
				->setParameter('ids', $ids);
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			$products[$row['id']] = [
				'id'            => $row['id'],
				'name'          => $row['name'],
				'price'         => (float) $row['price'],
				'retailPrice'   => (float) $row['retailPrice'],
				'localQuantity' => (int) $row['quantity'],
			];
		}


		$codes = [];
		$qb    = $this->em->getRepository(ProductSupplier::class)->createQueryBuilder('ps')
			->select('IDENTITY(ps.product) as id, ps.code, s.name')
			->innerJoin('ps.supplier', 's');

		if (!empty($ids)) {
			$qb->andWhere('ps.product IN (:ids)')
				->setParameter('ids', $ids);
		}

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			if (!empty($enabledSuppliers) && !in_array($row['name'], $enabledSuppliers)) {
				continue;
			}

			$codes[$row['name'] . '_' . $row['code']] = $row['id'];
		}

		$data = [];
		foreach ($this->priceCheck->getPrices() as $key => $row) {
			$product = $products[$codes[$key]] ?? null;

			if (!$product) {
				continue;
			}

			$id = $product['id'];

			if (!isset($data[$id])) {
				$data[$id] = [
					'id'            => $id,
					'name'          => $product['name'],
					'price'         => $product['price'],
					'retailPrice'   => $product['retailPrice'],
					'codes'         => [],
					'localQuantity' => $product['localQuantity'],
				];
			}

			$tmp                  = explode('_', $key);
			$vat                  = $row['vat'] ?: 21;
			$purchasePrice        = $row['purchasePrice'] * (((float) $vat + 100) / 100);
			$diff                 = round($product['price'] / $purchasePrice * 100, 2) - 100;
			$data[$id]['codes'][] = [
				'supplier'      => $tmp[0],
				'code'          => $row['supplierCode'],
				'purchasePrice' => $purchasePrice,
				'diff'          => $diff,
			];

			if (!isset($data[$id]['diff']) || $data[$id]['diff'] > $diff) {
				$data[$id]['diff'] = $diff;
			}
		}

		$products = null;
		$codes    = null;

		usort($data, fn($a, $b) => $a['diff'] <=> $b['diff']);

		return $data;
	}
}
