<?php declare(strict_types = 1);

namespace Import\AdminModule\Model\Readers;

use Core\Model\Helpers\Arrays;
use Core\Model\Helpers\Strings;
use Exception;
use Import\Model\ImportConfig;
use SimpleXMLElement;

class XmlReader implements IReader
{
	/** @var SimpleXMLElement[] */
	protected ?array  $cachedRawData = null;
	protected ?string $xPath         = null;

	public function __construct(protected ?string $fileContent)
	{
	}

	public function setXpath(mixed $xPath): self
	{
		$this->xPath = $xPath;

		return $this;
	}

	public function getKeys(): array
	{
		$keys = [];

		foreach ($this->getRawData() as $row) {
			if (!is_array($row)) {
				$keys[$row] = $row;
			} else {
				foreach (array_keys($row) as $k) {
					$keys[$k] = $k;
				}
			}
		}

		return $keys;
	}

	public function getValues(): array
	{
		$result = [];

		foreach ($this->getRawData() as $row) {
			foreach ($row as $rk => $rv) {
				if (is_string($rv) || is_numeric($rv)) {
					$result[$rk][] = $rv;
				}
			}
		}

		foreach ($result as $k => $v) {
			if (!is_numeric($v[0]) || in_array(Strings::lower($k), ['brand', 'manufacturer'])) {
				$result[$k] = array_unique($v);
				asort($result[$k]);
			} else {
				$result[$k] = array_chunk(array_unique($v), 150)[0];
			}
		}

		return $result;
	}

	protected function getRawData(): ?array
	{
		ini_set('memory_limit', ImportConfig::load('xmlReaderMemoryLimit'));
		if (!$this->cachedRawData) {
			$data = $this->fileContent ? simplexml_load_string(
				$this->fileContent,
				"SimpleXMLElement",
				LIBXML_NOCDATA | LIBXML_PARSEHUGE,
			) : null;

			if (!$data) {
				return [];
			}

			if ($this->xPath) {
				try {
					$newData = $data->xpath($this->xPath)[0];
					$data    = $newData;
				} catch (Exception) {
				}
			}
			$data = json_encode($data, JSON_THROW_ON_ERROR);
			$data = json_decode($data, true, 512, JSON_THROW_ON_ERROR);

			if (!is_bool($data) && (is_countable($data) ? count($data) : 0) == 1) {
				$data = array_values($data)[0];
			}
			if (isset($data['SHOPITEM'])) {
				$data = $data['SHOPITEM'];
			}
			if (isset($data['ShopItem'])) {
				$data = $data['ShopItem'];
			}

			if (isset($data['@attributes'])) {
				$data = end($data);
			}

			foreach ($data as &$row) {
				if (isset($row['@attributes'])) {
					$row += $row['@attributes'];
					unset($row['@attributes']);
				}
			}

			$this->cachedRawData = $data;
			$data                = null;
			$this->fileContent   = null;
		}

		return $this->cachedRawData;
	}

	/**
	 * TODO vylepšit
	 */
	public function getData(): array
	{
		$rawData = $this->getRawData();
		$data    = [];

		foreach ($rawData as $row) {
			$r = [];

			foreach ($row as $k => $v) {
				if (is_array($v) && empty($v)) {
					continue;
				}

				if (Arrays::strposa(
						strtolower((string) $k),
						['product', 'desc', 'url', 'img', 'manufa', 'categorytext', 'item', 'text'],
					)
					&& !is_array($v)) {
					$v = trim((string) $v);
				} else if (Arrays::strposa(strtolower((string) $k), ['price', 'vat'])) {
					$r['vat_original'] = $v;
					$v                 = str_replace(',', '.', (string) $v);
					$v                 = number_format((float) $v, 2, '.', '');
				} else if (is_array($v)) {
					$v = $this->loopArray($v);
				}

				$r[$k] = $v;
			}
			$data[] = $r;
		}
		$rawData = null;

		return $data;
	}

	private function loopArray(array|string $arr): array|string
	{
		$oArr = $arr;
		$arr  = (array) $arr;
		if (empty($arr)) {
			return (string) $oArr;
		}

		$paramKeyNames   = ['PARAM_NAME', 'PARAM', 'NAME'];
		$paramValueNames = ['VAL', 'VALUE'];
		$params          = [];

		foreach ($paramKeyNames as $keyName) {
			if (isset($arr[$keyName])) {
				$valueName = null;
				foreach ($paramValueNames as $vName) {
					if (isset($arr[$vName])) {
						$valueName = $vName;
					}
				}

				if ($valueName) {
					if (is_array($arr[$keyName])) {
						foreach ($arr[$keyName] as $i => $key) {
							$params[trim((string) $key)] = trim((string) $arr[$valueName][$i]);
						}
					} else if (is_string($arr[$keyName]) || is_numeric($arr[$keyName])) {
						$params[(string) $arr[$keyName]] = (string) $arr[$valueName];
					}
				}
			}
		}

		$tmp = [];
		if (empty($params)) {
			foreach ($arr as $key => $value) {
				if (is_array($value)) {
					$tmp[$key] = $this->loopArray($value);
				} else if (is_string($value) || is_numeric($value)) {
					$tmp[$key] = $value;
				}
			}
		}

		foreach ($tmp as $key => $value) {
			if (is_array($value) && count($value) == 1) {
				$params[key($value)] = $value[key($value)];
			} else {
				$params[$key] = $value;
			}
		}

		$arr = null;

		return $params;
	}

}
