<?php declare(strict_types = 1);

namespace Import\Console;

use Doctrine;
use Doctrine\ORM\Tools\Console\Command\SchemaTool\AbstractCommand;
use Exception;
use Import\AdminModule\Model\Imports;
use Import\AdminModule\Model\ProcessingTypes;
use Nette\DI\Attributes\Inject;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class Run extends AbstractCommand
{
	#[Inject]
	public Imports $imports;

	#[Inject]
	public ProcessingTypes $processingTypes;

	protected function configure(): void
	{
		$this->setName('imports:run')
			->addOption('id', null, InputOption::VALUE_REQUIRED)
			->setDescription('Run import');
	}

	protected function executeSchemaCommand(
		InputInterface                $input,
		OutputInterface               $output,
		Doctrine\ORM\Tools\SchemaTool $schemaTool,
		array                         $metadatas,
		SymfonyStyle                  $ui,
	): int
	{
		$output->writeln('<fg=white;bg=#007bff>IMPORT - Run</>');
		try {
			ini_set('memory_limit', '2G');
			$import = $this->imports->getAsArray((int) $input->getOption('id'));

			if ($import && $import['isActive']) {

				$processingType = $this->processingTypes->getType($import['type'], $import['processingType']);
				if ($processingType && $import['id']) {
					$importEntity = $this->imports->get($import['id']);

					if (!$importEntity) {
						throw new Exception('Import entity not found: ' . $import['id']);
					}

					$fileContent = $this->imports->getFileContent($importEntity);
					$fileData    = $fileContent ? $this->imports->getFileData($fileContent, $importEntity) : [];

					$responseData = $processingType->run($importEntity, $fileData);
					$this->imports->updateLastRun($importEntity->getId(), $importEntity->url);
					$this->imports->addHistory($importEntity->getId(), $responseData);

					$output->writeLn('<info>[OK] - Import done</info>');

					return Command::SUCCESS;
				}

				$output->writeln('<error>[ERROR] - Processing type not found</error>');
			} else {
				$output->writeln('<error>[ERROR] - Not found or is not active</error>');
			}
		} catch (Exception $e) {
			$output->writeln('<error>[ERROR] - ' . $e->getMessage() . '</error>');
		}

		return Command::FAILURE;
	}
}
