<?php declare(strict_types = 1);

namespace Import\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Import\AdminModule\Model\Imports;
use Import\AdminModule\Model\ImportTypes;
use Import\AdminModule\Model\ProcessingTypes;
use Import\Model\Entities\Import;
use Nette\Utils\ArrayHash;

class ImportForm extends BaseControl
{
	/** @var Import */
	protected $import;

	/** @var Imports */
	protected $importsService;

	/** @var ImportTypes */
	protected $importTypes;

	/** @var ProcessingTypes */
	protected $processingTypes;

	public function __construct(Imports $imports, ImportTypes $importTypes, ProcessingTypes $processingTypes)
	{
		$this->importsService  = $imports;
		$this->importTypes     = $importTypes;
		$this->processingTypes = $processingTypes;
	}

	public function render(): void
	{
		$this->template->import = $this->import;
		$this->template->render($this->getTemplateFile());
	}

	public function createComponentForm(): BaseForm
	{
		$form = $this->createForm();

		$form->addText('title', 'import.import.title');
		$form->addTextArea('description', 'import.import.description');
		$form->addBool('isActive', 'import.import.isActive');
		$form->addText('url', 'import.import.url');
		$form->addBool('allowUpload', 'import.import.allowUpload')->setDefaultValue(0);
		$form->addSelect('type', 'import.import.type', $this->importTypes->getSelectArr());

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		try {
			if ($this->import) {
				$import = $this->import;

				$import->processingType = $values->processingType;

				$importType = $this->importTypes->getType($values->type);
				if ($importType) {
					$import->syncOpts = $importType->processSubmitSyncOpts($values->syncOpts);
					$import->data     = $importType->processSubmitData($values->data);
				}
			} else
				$import = new Import($values->title, $values->type);

			$import->title       = $values->title;
			$import->description = $values->description;
			$import->isActive    = $values->isActive ? 1 : 0;
			$import->url         = $values->url;
			$import->allowUpload = $values->allowUpload ? 1 : 0;
			$import->type        = $values->type;

			$flashMessage = $import->getId() ? 'import.import.edited' : 'import.import.added';

			$this->em->persist($import)->flush();
			$form->addCustomData('importId', $import->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}

		return true;
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setImport(int $id): void
	{
		$this->import = $this->importsService->get($id);

		if ($this->import) {
			$form   = $this['form'];
			$import = $this->import;

			$form->addSelect('processingType', 'import.import.processingType', $this->processingTypes->getSelectArr($import->type))
				->setDefaultValue('pairing');

			// TODO přidat do nastavení formuláře, teď se vše tahá z url, ale ta není vždy vyplněná.
			$importFileContent = $this->importsService->getFileContent($import);
			$importFileData    = $this->importsService->getFileData($importFileContent, $import);

			$importType = $this->importTypes->getType($import->type);
			if ($importType) {
				$form->addComponent($importType->getFormSyncOpts($importFileData['keys'], $importFileData['values'], $import->getSyncOpts()), 'syncOpts');
				$form->addComponent($importType->getFormContainer($importFileData['keys'], $importFileData['values'], $import->data), 'data');
			}

			$this->onAnchor[] = function() use ($importFileData) {
				$this->template->importFileData = $importFileData;
			};

			$form->setDefaults([
				'title'       => $import->title,
				'description' => $import->description,
				'isActive'    => $import->isActive,
				'url'         => $import->url,
				'allowUpload' => $import->allowUpload,
			]);

			if ($import->type && array_key_exists($import->type, $form['type']->getItems()))
				$form['type']->setDefaultValue($import->type);
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
