<?php declare(strict_types = 1);

namespace InfoPanels\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use InfoPanels\AdminModule\Model\MessagesService;
use InfoPanels\AdminModule\Model\PanelsService;
use InfoPanels\Model\Entities\Message;
use Nette\Http\IResponse;
use Nette\Utils\ArrayHash;

class MessageForm extends TemplateReaderControl
{
	/** @var Message */
	public $entity;

	/** @var MessagesService */
	protected $messagesService;

	/** @var PanelsService */
	protected $panelsService;

	public function __construct(PanelsService $panelsService, MessagesService $messagesService)
	{
		$this->panelsService   = $panelsService;
		$this->messagesService = $messagesService;
	}

	public function render()
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm()
	{
		$this->templateReader->setTemplatesDir($this->messagesService->getTemplatesDir());
		$this->templateReader->setTranslateKey('infoPanels.messageTemplate');

		$form = $this->createForm();

		$templates = $this->templateReader->getTemplates();

		$panels = [];
		foreach ($this->panelsService->getEr()->findBy([], ['title' => 'ASC']) as $p) {
			$panels[$p->getId()] = $p->title;
		}

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addSelect('panel', 'infoPanels.messageForm.panel', $panels)->setRequired();
		$form->addSelect('template', $this->t('default.templates'), $templates)->setPrompt($this->t('default.choose'))
			->setTranslator(null)->setRequired();
		$form->addComponent(new BaseContainer(), 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		try {
			$panel = $this->panelsService->getReference($values->panel);

			if ($this->entity) {
				$message      = $this->entity;
				$flashMessage = 'infoPanels.messageForm.edited';
			} else {
				$message      = new Message($values->title, $panel, $values->template);
				$flashMessage = 'infoPanels.messageForm.added';
			}

			$message->title       = $values->title;
			$message->isPublished = $values->isPublished;
			$message->panel       = $panel;
			$message->template    = $values->template;
			$message->setTexts($form->getComponent('component')->getValues(true));

			$this->em->persist($message)->flush();
			$form->addCustomData('messageId', $message->getId());
			$this->getPresenter()->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$form->addError($e->getMessage());

			return false;
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	public function setMessage($id)
	{
		$this->entity = $this->messagesService->get($id);

		if ($this->entity) {
			$form = $this['form'];
			$e    = $this->entity;
			$form->setDefaults([
				'title'       => $e->title,
				'isPublished' => $e->isPublished,
			]);

			if ($e->panel && array_key_exists($e->panel->getId(), $form['panel']->getItems()))
				$form['panel']->setDefaultValue($e->panel->getId());


			if ($e->template && array_key_exists($e->template, $form['template']->getItems())) {
				$form['template']->setDefaultValue($e->template);

				if ($this->httpRequest->getQuery('do') != 'messageForm-loadInputs') {
					$this->templateReader->loadTemplateComponents($form['component'], $this->httpRequest->getPost('template') ?: $e->template);
					$this->templateReader->setDefaults($form['component'], $e, $e->template);
				}
			}
		} else
			$this->getPresenter()->error(null, IResponse::S404_NOT_FOUND);
	}
}
