<?php declare(strict_types = 1);

namespace InfoPanels\FrontModule\Model;

use Core\Model\Helpers\BaseFrontEntityService;
use Doctrine\ORM\QueryBuilder;
use InfoPanels\FrontModule\Model\Dao\Message;
use InfoPanels\Model\Entities\Panel;
use InfoPanels\Model\PanelsCache;
use Nette\Caching\Cache;

class PanelsService extends BaseFrontEntityService
{
	protected $entityClass = Panel::class;

	public function __construct(
		protected PanelsCache $panelsCache,
	)
	{
	}

	/**
	 * @return QueryBuilder
	 */
	protected function setPublishedQueryBuilder(QueryBuilder $qb)
	{
		$qb->andWhere('p.isPublished = 1');

		if (in_array('m', $qb->getAllAliases())) {
			$qb->andWhere('m.isPublished = 1');
		}

		return $qb;
	}

	/**
	 * @param int|string|array $id
	 *
	 * @return Dao\Panel|Dao\Panel[]|null
	 */
	public function get($id)
	{
		$ids   = [];
		$keys  = [];
		$data  = [];
		$where = [];
		foreach (is_array($id) ? $id : [$id] as $v) {
			$ids[]  = $v;
			$keys[] = 'panel/' . $v;
		}

		foreach ($this->panelsCache->getCache()->bulkLoad($keys) as $k => $v) {
			$tmp = explode('/', $k);

			if ($v !== null) {
				$data[$tmp[1]] = $v;
			} else {
				$where[]       = $tmp[1];
				$data[$tmp[1]] = [];
			}
		}

		if ($where !== []) {
			$qb = $this->getEr()->createQueryBuilder('p')->addSelect('m')
				->join('p.messages', 'm');

			$whereIds = [];
			$wherePos = [];

			foreach ($where as $v) {
				if (is_numeric($v)) {
					$whereIds[] = $v;
				} else {
					$wherePos[] = $v;
				}
			}

			foreach ($wherePos as $k => $v) {
				$qb->orWhere('p.positions LIKE :pos' . $k)
					->setParameter('pos' . $k, $v);
			}

			if ($whereIds !== []) {
				$qb->orWhere('p.id IN (:ids)')
					->setParameter('ids', $ids);
			}

			$qb = $this->setPublishedQueryBuilder($qb);

			foreach ($qb->getQuery()->getArrayResult() as $row) {
				/** @var array $row */
				$panel = new Dao\Panel($row['id'], $row['title']);

				foreach ($row['messages'] as $v) {
					$message           = new Message($v['id'], $v['title'], $v['template']);
					$message->modified = $v['modified'];
					$message->texts    = $v['texts'];

					$panel->messages[$message->id] = $message;
				}

				$data[$row['id']] = $panel;
				$this->panelsCache->getCache()->save('panel/' . $row['id'], $panel, [Cache::TAGS => ['infopanels']]);

				foreach (explode(',', (string) $row['positions']) as $pos) {
					$data[$pos][$panel->getId()] = $panel;
				}
			}

			foreach ($where as $v) {
				if (empty($data[$v])) {
					$this->panelsCache->getCache()->save('panel/' . $v, [], [Cache::TAGS => ['infopanels']]);
				}
			}

			foreach ($data as $k => $panels) {
				if (is_string($k)) {
					$this->panelsCache->getCache()->save('panel/' . $k, $panels, [Cache::TAGS => ['infopanels']]);
				}
			}
		}

		if (count($ids) === 1) {
			return $data[$ids[0]];
		}

		$result = [];
		foreach ($ids as $v) {
			if (isset($data[$v])) {
				$result[$v] = $data[$v];
			}
		}

		return $result;
	}
}
