<?php declare(strict_types = 1);

namespace InfoPanels\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\DataGrid\BaseDataGrid;
use InfoPanels\AdminModule\Model\MessagesService;
use InfoPanels\AdminModule\Model\PanelsService;
use Nette\Application\AbortException;

class MessagesGrid extends BaseControl
{
	protected MessagesService $messagesService;
	protected PanelsService   $panelsService;

	public function __construct(
		MessagesService $messagesService,
		PanelsService   $panelsService
	)
	{
		$this->messagesService = $messagesService;
		$this->panelsService   = $panelsService;
	}

	public function render(): void
	{
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Handle
	 */

	/**
	 * @param int|string $id
	 *
	 * @throws AbortException
	 */
	public function handleDelete($id): void
	{
		$presenter = $this->presenter;
		if ($this->messagesService->remove($id)) {
			$presenter->flashMessageSuccess('default.removed');
		} else {
			$presenter->flashMessageDanger('default.removeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->reload();
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentGrid(): BaseDataGrid
	{
		$grid = $this->createGrid();

		$qb = $this->messagesService->getEr()->createQueryBuilder('m')
			->leftJoin('m.panel', 'p')->addSelect('p');
		$grid->setDataSource($qb);

		// Columns
		$grid->addColumnLink('title', 'default.title', 'Default:editMessage');
		$grid->addColumnStatus('isPublished', 'default.isPublished')->setAlign('center')
			->addOption(1, 'default.publish')->setIcon('check')->setClass('btn-success')->setShowTitle(false)->endOption()
			->addOption(0, 'default.unPublish')->setIcon('times')->setClass('btn-danger')->setShowTitle(false)->endOption()
			->onChange[] = [$this, 'gridPublishChange'];
		$grid->addColumnText('panel', 'infoPanels.messagesGrid.panel', 'panel.title');

		// Filter
		$grid->addFilterText('title', '');
		$grid->addFilterSelect('isPublished', '', [
			'' => '',
			0  => $this->t('default.no'),
			1  => $this->t('default.yes')]);
		$panels = ['' => ''];
		foreach ($this->panelsService->getEr()->findBy([], ['title' => 'ASC']) as $panel) {
			$panels[$panel->getId()] = $panel->title;
		}
		$grid->addFilterSelect('panel', '', $panels);

		// Actions
		$grid->addAction('edit', '', 'Default:editMessage')->setIcon('edit')->setBsType('primary');
		$grid->addAction('delete', '', 'delete!')->setConfirm('default.reallyDelete')->setIcon('times')->setBsType('danger')->addClass('ajax');

		// Columns prototype
		$grid->getColumn('isPublished')->getElementPrototype('th')->class[] = 'w1';

		return $grid;
	}

	/**
	 * @param int      $id
	 * @param int|bool $newStatus
	 *
	 * @throws AbortException
	 */
	public function gridPublishChange($id, $newStatus): void
	{
		$presenter = $this->presenter;

		if ($this->messagesService->setPublish($id, $newStatus)) {
			$presenter->flashMessageSuccess('default.publishChanged');
		} else {
			$presenter->flashMessageDanger('default.publishChangeFailed');
		}

		if ($presenter->isAjax()) {
			$this['grid']->redrawItem($id);
			$presenter->redrawControl('flashes');
		} else {
			$presenter->redirect('this');
		}
	}
}
