<?php declare(strict_types = 1);

namespace InfoPanels\AdminModule\Components;

use Core\Model\TemplateReader\TemplateReaderControl;
use Core\Model\UI\Form\BaseContainer;
use Core\Model\UI\Form\BaseForm;
use Exception;
use InfoPanels\AdminModule\Model\MessagesService;
use InfoPanels\AdminModule\Model\PanelsService;
use InfoPanels\Model\Entities\Message;
use InfoPanels\Model\Entities\Panel;
use InfoPanels\Model\PanelsCache;
use Nette\Application\BadRequestException;
use Nette\Utils\ArrayHash;

class MessageForm extends TemplateReaderControl
{
	/** @var Message|null */
	public $entity;

	public function __construct(
		protected PanelsService   $panelsService,
		protected MessagesService $messagesService,
		protected PanelsCache     $panelsCache,
	)
	{
	}

	public function render(): void
	{
		$this->template->componentStructure = $this->templateReader->getComponentStructure();
		$this->template->thisForm           = $this['form'];

		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ==================  Components
	 */

	protected function createComponentForm(): BaseForm
	{
		$this->templateReader->setTemplatesDir($this->messagesService->getTemplatesDir());
		$this->templateReader->setTranslateKey('infoPanels.messageTemplate');

		$form = $this->createForm();

		$templates = $this->templateReader->getTemplates();

		$panels = [];
		foreach ($this->panelsService->getEr()->findBy([], ['title' => 'ASC']) as $p) {
			/** @var Panel $p */
			$panels[$p->getId()] = $p->title;
		}

		$form->addText('title', 'default.title')->setRequired()->setMaxLength(255);
		$form->addBool('isPublished', 'default.isPublished')->setDefaultValue(1);
		$form->addSelect('panel', 'infoPanels.messageForm.panel', $panels)->setRequired();
		$form->addSelect('template', $this->t('default.templates'), $templates)->setPrompt($this->t('default.choose'))
			->setTranslator(null)->setRequired();
		$form->addComponent(new BaseContainer, 'component');

		$form->addSaveCancelControl();

		$form->onSuccess[] = $this->formSuccess(...);

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): void
	{
		try {
			$panel = $this->panelsService->get($values->panel);
			if (!$panel) {
				throw new BadRequestException('Panel not found');
			}

			if ($this->entity) {
				$message      = $this->entity;
				$flashMessage = 'infoPanels.messageForm.edited';
			} else {
				$message      = new Message($values->title, $panel, $values->template);
				$flashMessage = 'infoPanels.messageForm.added';
			}

			$message->title       = $values->title;
			$message->isPublished = $values->isPublished;
			$message->panel       = $panel;
			$message->template    = $values->template;

			$httpData = (array) $form->getHttpData();
			$message->setTexts((array) ($httpData['component'] ?? []));

			$this->em->persist($message);
$this->em->flush();
			$form->addCustomData('messageId', $message->getId());
			$this->presenter->flashMessageSuccess($flashMessage);
		} catch (Exception $e) {
			$form->addError($e->getMessage());
		}
	}

	/*******************************************************************************************************************
	 * ==================  GET / SET
	 */

	/**
	 * @param int|string $id
	 *
	 * @throws BadRequestException
	 */
	public function setMessage($id): void
	{
		$this->entity = $this->messagesService->get($id);

		if ($this->entity) {
			$form = $this['form'];
			$e    = $this->entity;
			$form->setDefaults([
				'title'       => $e->title,
				'isPublished' => $e->isPublished,
			]);

			if ($e->panel->getId() && array_key_exists($e->panel->getId(), $form->getComponent('panel')->getItems())) {
				$form->getComponent('panel')->setDefaultValue($e->panel->getId());
			}

			if ($e->template && array_key_exists($e->template, $form->getComponent('template')->getItems())) {
				$form->getComponent('template')->setDefaultValue($e->template);
				if ($this->httpRequest->getQuery('do') !== 'messageForm-loadInputs') {
					/** @var BaseContainer $container */
					$container = $form['component'];

					/** @var string $template */
					$template = $this->httpRequest->getPost('template') ?: $e->template;

					$this->templateReader->loadTemplateComponents($container, $template, $e->getTexts());
					$this->templateReader->setDefaults($container, $e, $e->template);
				}
			}
		} else {
			$this->presenter->error();
		}
	}

}
