<?php declare(strict_types = 1);

namespace Intranet\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Intranet\Model\Entities\File;
use Intranet\Model\Entities\Folder;
use Intranet\Model\FilesService;
use Intranet\Model\FoldersService;
use Nette\Application\ForbiddenRequestException;

class DocsStructure extends BaseControl
{
	/** @var @persistent */
	public $folderId;

	protected FoldersService     $foldersService;
	protected FilesService       $filesService;
	protected IFolderFormFactory $folderFormFactory;

	protected ?Folder $folder = null;

	private $structure;

	/** @var File[]|null */
	protected $files;

	public function __construct(
		FoldersService     $foldersService,
		FilesService       $filesService,
		IFolderFormFactory $folderFormFactory
	)
	{
		$this->foldersService    = $foldersService;
		$this->filesService      = $filesService;
		$this->folderFormFactory = $folderFormFactory;
	}

	public function render(): void
	{
		$this->template->currentFolder = $this->folder;
		$this->template->folders       = $this->getStructure();
		$this->template->files         = $this->getFiles();
		$this->template->canManage     = $this->getPresenter()->getUser()->isAllowed('Intranet:Admin', 'manage');
		$this->template->render($this->getTemplateFile());
	}

	protected function attached($presenter): void
	{
		parent::attached($presenter);
		$this->setFolder($this->getParameter('folderId'));
	}

	/*******************************************************************************************************************
	 * =================  Handle
	 */

	public function handleFolder($folderId)
	{
		$this->setFolder($folderId);
		if ($this->folder) {
			$this->getPresenter()['header']->setTitle($this->t('intranet.structure.openedFolder', null, ['folder' => $this->folder->title]));
		}
	}

	public function handleDeleteFolder($id): void
	{
		$presenter = $this->presenter;

		if (!$presenter->getUser()->isAllowed('Intranet:Admin', 'manage')) {
			$presenter->flashMessageDanger('default.accessForbidden');
			$presenter->redrawControl('flashes');
		} else {
			$removed = false;
			try {
				if ($removed = $this->foldersService->remove($id)) {
					$presenter->flashMessageSuccess('intranet.folder.folderRemoved');
				} else {
					$presenter->flashMessageDanger('intranet.folder.folderRemoveFailed');
				}
			} catch (\Exception $e) {
				$presenter->flashMessageDanger('intranet.folder.' . $e->getMessage());
			}

			$presenter->redrawControl('flashes');

			if ($removed) {
				$this->redrawControl('folders');
			}
		}
	}

	public function handleDeleteFile($id): void
	{
		$presenter = $this->presenter;

		if (!$presenter->getUser()->isAllowed('Intranet:Admin', 'manage')) {
			$presenter->flashMessageDanger('default.accessForbidden');
			$presenter->redrawControl('flashes');
		} else {
			if ($this->filesService->remove($id)) {
				$presenter->flashMessageSuccess('intranet.file.fileRemoved');
			} else {
				$presenter->flashMessageDanger('intranet.file.fileRemoveFailed');
			}

			$presenter->redrawControl('flashes');
			$this->redrawControl('files');
		}
	}

	public function handleEditFolder(int $id): void
	{
		$presenter = $this->presenter;
		if (!($folder = $this->foldersService->get($id)) || !$presenter->getUser()->isAllowed('Intranet:Admin', 'manage')) {
			throw new ForbiddenRequestException();
		}

		$this->folderId             = $id;
		$this->template->modalTitle = $this->t('intranet.title.folderEdit') . ': ' . $folder->title;
		$this->template->modal      = 'folderForm';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * =================  Get/Set
	 */

	public function setFolder($folderId)
	{
		$this->folderId = $folderId;
		$this->folder   = $folderId ? $this->foldersService->get($folderId) : null;
	}

	public function getStructure()
	{
		if ($this->structure === null) {
			$this->structure = $this->folder ? $this->folder->getChildrens() : $this->foldersService->getTopLvl();
		}

		return $this->structure;
	}

	public function getFiles()
	{
		if ($this->files === null) {
			$this->files = $this->folder ? $this->folder->getPublishedFiles() : $this->filesService->getFilesWithoutFolder();
		}

		foreach ($this->files as $k => $v) {
			$data                 = base64_encode(serialize([
				'file' => $v->getFilePath(),
			]));
			$this->files[$k]->url = $this->getPresenter()->link(':Core:Admin:FileResponse:', [$data]);
		}

		return $this->files;
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentFolderForm(): FolderForm
	{
		$control = $this->folderFormFactory->create();

		if ($this->folderId) {
			$control->setFolder((int) $this->folderId);
		}

		$control['form']->onSuccessSave[] = function(BaseForm $form) {
			$folder       = $this->foldersService->get((int) $form->getCustomData('folderId'));
			$this->folder = $folder->getParent();
			if ($this->folder) {
				$this->folderId = $this->folder->getId();
			}

			$this->template->modalTitle = $this->t('intranet.title.folderEdit') . ': ' . $folder->title;
			$this->template->modal      = 'folderForm';
			$this->redrawControl('modal');
			$this->redrawControl('flashes');
			$this->redrawControl('folders');
		};

		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$folder       = $this->foldersService->get((int) $form->getCustomData('folderId'));
			$this->folder = $folder->getParent();
			if ($this->folder) {
				$this->folderId = $this->folder->getId();
			}

			$this->getPresenter()->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this->redrawControl('folders');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}
}
