<?php declare(strict_types = 1);

namespace Intranet\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Intranet\Model\Entities\File;
use Intranet\Model\FilesService;
use Intranet\Model\FoldersService;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class FileForm extends BaseControl
{
	protected FilesService   $filesService;
	protected FoldersService $foldersService;

	protected ?File $file = null;

	public function __construct(
		FilesService   $filesService,
		FoldersService $foldersService
	)
	{
		$this->filesService   = $filesService;
		$this->foldersService = $foldersService;
	}

	public function render(): void
	{
		if (!$this->getPresenter()->getUser()->isAllowed('Intranet:Admin', 'manage')) {
			return;
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$folders = ['' => $this->t('intranet.fileForm.chooseFolder')] + $this->foldersService->getFoldersForSelect();

		$form->addText('title', 'intranet.fileForm.title')->setRequired();
		$form->addSelect('folder', 'intranet.fileForm.folder', $folders);
		if (!$this->file) {
			$form->addUpload('file', 'intranet.fileForm.file');
		}

		$form->addHidden('id');
		$form->addSaveCancelControl('saveControl');
		$form['saveControl']->removeComponent($form['saveControl']['save']);

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values)
	{
		$presenter = $this->presenter;
		try {
			$upload = $values->file;
			if ($this->file) {
				$file = $this->file;
			} else if ($upload && $upload->isOk()) {
				$fileExt  = pathinfo($upload->getSanitizedName(), PATHINFO_EXTENSION);
				$fileName = str_replace('.' . $fileExt, '', $upload->getSanitizedName()) . uniqid((string) rand(0, 20)) . '.' . $fileExt;
				$upload->move(File::FILE_PATH . '/' . $fileName);
				$file = new File($values->title, $fileName);
			} else {
				$form->addError('intranet.fileForm.fileUploadError');

				return false;
			}

			$file->title = $values->title;
			//			$file->status = $values->published;
			$file->setFolder($values->folder && $values->folder != '' ? $this->foldersService->getReference($values->folder) : null);

			$flashMessage = $file->getId() ? 'intranet.fileForm.fileEdited' : 'intranet.fileForm.fileAdded';

			$this->em->persist($file);
			$this->em->flush();

			$form['id']->setValue($file->getId());

			$form->addCustomData('fileId', $file->getId());
			$presenter->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$presenter->flashMessageDanger($e->getMessage());
			$presenter->redrawControl('flashes');

			return false;
		}

		return true;
	}

	public function setFile($file): self
	{
		if ($file instanceof File) {
			$this->file = $file;
		} else {
			$file = $this->filesService->get($file);
		}

		if (!$file) {
			throw new InvalidArgumentException();
		}

		$this->file = $file;
		$form       = $this['form'];

		$form->setDefaults([
			'title' => $file->title,
		]);

		if ($file->getParent() && array_key_exists($file->getFolder()->getId(), $form['folder']->getItems())) {
			$form['folder']->setDefaultValue($file->getFolder()->getId());
		}

		return $this;
	}
}
