<?php declare(strict_types = 1);

namespace Intranet\AdminModule\Components;

use Core\Model\UI\BaseControl;
use Core\Model\UI\Form\BaseForm;
use Intranet\Model\Entities\Folder;
use Intranet\Model\FoldersService;
use Nette\InvalidArgumentException;
use Nette\Utils\ArrayHash;

class FolderForm extends BaseControl
{
	protected FoldersService $foldersService;

	protected ?Folder $folder = null;

	public function __construct(
		FoldersService $foldersService
	)
	{
		$this->foldersService = $foldersService;
	}

	public function render(): void
	{
		if (!$this->getPresenter()->getUser()->isAllowed('Intranet:Admin', 'manage')) {
			return;
		}

		$this->template->render($this->getTemplateFile());
	}

	protected function createComponentForm(): BaseForm
	{
		$form = $this->createForm();
		$form->setAjax();

		$folders = ['' => $this->t('intranet.folderForm.chooseParent')] + $this->foldersService->getFoldersForSelect();

		$form->addText('title', 'intranet.folderForm.title')->setRequired();
		$form->addSelect('parent', 'intranet.folderForm.parent', $folders);
		//		$form->addBool('published', 'intranet.folderForm.published')->setDefaultValue(1);

		$form->addHidden('id');
		$form->addSaveCancelControl('saveControl');

		$form->onSuccess[] = [$this, 'formSuccess'];

		return $form;
	}

	public function formSuccess(BaseForm $form, ArrayHash $values): bool
	{
		$presenter = $this->presenter;
		try {
			$folder = null;
			if ($this->folder) {
				$folder = $this->folder;
			} else if ($values->id) {
				$folder = $this->foldersService->get($values->id);
			}

			if (!$folder) {
				$folder = new Folder($values->title);
			}

			$folder->title = $values->title;
			//			$folder->status = $values->published;
			$folder->setParent($values->parent && $values->parent != '' ? $this->foldersService->get($values->parent) : null);

			$flashMessage = $folder->getId() ? 'intranet.folderForm.folderEdited' : 'intranet.folderForm.folderAdded';

			$this->em->persist($folder);
			$this->em->flush();

			$form['id']->setValue($folder->getId());

			$form->addCustomData('folderId', $folder->getId());
			$presenter->flashMessageSuccess($flashMessage);
		} catch (\Exception $e) {
			$presenter->flashMessageDanger($e->getMessage());
			$presenter->redrawControl('flashes');

			return false;
		}

		return true;
	}

	public function setFolder($folder): self
	{
		if ($folder instanceof Folder) {
			$this->folder = $folder;
		} else {
			$folder = $this->foldersService->get($folder);
		}

		if (!$folder) {
			throw new InvalidArgumentException();
		}

		$this->folder = $folder;
		$form         = $this['form'];

		$form->setDefaults([
			'title' => $folder->title,
		]);

		if ($folder->getParent() && array_key_exists($folder->getParent()->getId(), $form['parent']->getItems())) {
			$form['parent']->setDefaultValue($folder->getParent()->getId());
		}

		return $this;
	}
}
