<?php declare(strict_types = 1);

namespace Intranet\AdminModule\Presenters;

use Core\Model\UI\Form\BaseForm;
use Intranet\AdminModule\Components\DocsStructure;
use Intranet\AdminModule\Components\FileForm;
use Intranet\AdminModule\Components\FolderForm;
use Intranet\AdminModule\Components\IDocsStructureFactory;
use Intranet\AdminModule\Components\IFileFormFactory;
use Intranet\AdminModule\Components\IFolderFormFactory;
use Intranet\Model\FoldersService;
use Nette\Application\ForbiddenRequestException;

class DefaultPresenter extends BasePresenter
{
	protected FoldersService $foldersService;

	public function __construct(
		FoldersService $foldersService
	)
	{
		$this->foldersService = $foldersService;
	}

	protected function startup()
	{
		parent::startup();
		$this->setHeader('intranet.title.intranet', 'fas fa-file-contract');
	}

	public function actionDefault(): void
	{
		if ($this->getUser()->isAllowed('Intranet:Admin', 'manage')) {
			$this['navigation']->setData(['header' => [
				[
					'title' => 'intranet.menu.newFolder',
					'link'  => $this->link('newFolder!'),
					'ico'   => 'folder',
					'class' => 'ajax',
				],
				[
					'title' => 'intranet.menu.newFile',
					'link'  => $this->link('newFile!'),
					'ico'   => 'file',
					'class' => 'ajax',
				],
			]]);
		}
	}

	public function handleNewFolder(): void
	{
		if (!$this->getUser()->isAllowed('Intranet:Admin', 'manage')) {
			throw new ForbiddenRequestException();
		}

		$this->template->modalTitle = $this->t('intranet.title.newFolder');
		$this->template->modal      = 'folderForm';
		$this->redrawControl('modal');
	}

	public function handleNewFile(): void
	{
		if (!$this->getUser()->isAllowed('Intranet:Admin', 'manage')) {
			throw new ForbiddenRequestException();
		}

		$this->template->modalTitle = $this->t('intranet.title.newFile');
		$this->template->modal      = 'fileForm';
		$this->redrawControl('modal');
	}

	/*******************************************************************************************************************
	 * ============================== Components
	 */

	protected function createComponentFolderForm(IFolderFormFactory $factory): FolderForm
	{
		$control = $factory->create();

		if ($this['docsStructure']->folderId) {
			$control['form']['parent']->setDefaultValue($this['docsStructure']->folderId);
		}

		$control['form']->onSuccessSave[] = function(BaseForm $form) {
			$folder = $this->foldersService->get((int) $form->getCustomData('folderId'));

			$this->template->modalTitle = $this->t('intranet.title.folderEdit') . ': ' . $folder->title;
			$this->template->modal      = 'folderForm';
			$this->redrawControl('modal');
			$this->redrawControl('flashes');
			$this->redrawControl('docsStructure');
		};

		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this->redrawControl('docsStructure');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentFileForm(IFileFormFactory $factory): FileForm
	{
		$control = $factory->create();

		if ($this['docsStructure']->folderId) {
			$control['form']['folder']->setDefaultValue($this['docsStructure']->folderId);
		}

		$control['form']->onSuccessSaveAndClose[] = function(BaseForm $form) {
			$this->payload->hideModal = true;
			$this->redrawControl('flashes');
			$this->redrawControl('docsStructure');
		};

		$control['form']['saveControl']->closeModalOnCancel();

		return $control;
	}

	protected function createComponentDocsStructure(IDocsStructureFactory $factory): DocsStructure
	{
		return $factory->create();
	}
}
