<?php declare(strict_types = 1);

namespace Intranet\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

/**
 * @ORM\Table(name="intranet__file")
 * @ORM\Entity
 */
class File
{
	use TId;

	public const STATUS_PUBLISHED = 1;
	public const STATUS_DISABLED  = 0;

	public const FILE_PATH = ROOT_DIR . '/intranet';

	/**
	 * @ORM\Column(name="title", type="string", length=255, nullable=false)
	 */
	public string $title;

	/**
	 * @Gedmo\SortablePosition
	 * @ORM\Column(name="position", type="integer")
	 */
	protected ?int $position = null;

	/**
	 * @ORM\Column(name="status", type="smallint", nullable=false, options={"default":1})
	 */
	public int $status = 1;

	/**
	 * @ORM\ManyToOne(targetEntity="Folder", inversedBy="files")
	 * @ORM\JoinColumn(name="folder_id", referencedColumnName="id", onDelete="RESTRICT", nullable=true)
	 */
	protected ?Folder $folder = null;

	/**
	 * @ORM\Column(name="file", type="string", length=255, nullable=false)
	 */
	protected string $file;

	public ?string $url = null;

	public function __construct(
		string $title,
		string $file
	)
	{
		$this->title  = $title;
		$this->status = 1;
		$this->file   = $file;
		$this->setPosition(-1);
	}

	public function getFolder(): ?Folder { return $this->folder; }

	public function setFolder(?Folder $folder = null): void
	{
		$this->folder = $folder;
	}

	public function getFilePath(): string { return self::FILE_PATH . '/' . $this->file; }

	public function getFileType(): ?string
	{
		$ext = strtolower(pathinfo($this->file, PATHINFO_EXTENSION));

		if (in_array($ext, ['jpg', 'jpeg', 'png', 'gif'])) {
			return 'image';
		} else if (in_array($ext, ['xls', 'xlsx'])) {
			return 'excel';
		} else if (in_array($ext, ['doc', 'docx'])) {
			return 'word';
		} else if (in_array($ext, ['zip', 'rar'])) {
			return 'archive';
		} else if ($ext === 'pdf') {
			return 'pdf';
		}

		return null;
	}

	public function setPosition(int $position): void
	{
		$this->position = $position;
	}

	public function getPosition(): ?int { return $this->position; }
}
