<?php declare(strict_types = 1);

namespace Intranet\Model\Entities;

use Core\Model\Entities\TId;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\Common\Collections\Criteria;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

/**
 * @Gedmo\Tree(type="nested")
 * @ORM\Table(name="intranet__folder")
 * @ORM\Entity(repositoryClass="FolderRepository")
 */
class Folder
{
	use TId;

	public const STATUS_PUBLISHED = 1;
	public const STATUS_DISABLED  = 0;

	/**
	 * @ORM\Column(name="title", type="string", length=255, nullable=false)
	 */
	public string $title;

	/**
	 * @ORM\Column(name="status", type="smallint", nullable=false, options={"default":1})
	 */
	public int $status = 1;

	/**
	 * @Gedmo\TreeLeft
	 * @ORM\Column(name="lft", type="integer")
	 */
	protected ?int $lft = null;

	/**
	 * @Gedmo\TreeLevel
	 * @ORM\Column(name="lvl", type="integer")
	 */
	protected ?int $lvl = null;

	/**
	 * @Gedmo\TreeRight
	 * @ORM\Column(name="rgt", type="integer")
	 */
	protected ?int $rgt = null;

	/**
	 * @Gedmo\TreeRoot
	 * @ORM\ManyToOne(targetEntity="Folder")
	 * @ORM\JoinColumn(name="tree_root", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?Folder $root = null;

	/**
	 * @Gedmo\TreeParent
	 * @ORM\ManyToOne(targetEntity="Folder", inversedBy="children")
	 * @ORM\JoinColumn(name="parent_id", referencedColumnName="id", onDelete="CASCADE")
	 */
	protected ?Folder $parent = null;

	/**
	 * @var Collection<Folder>
	 * @ORM\OneToMany(targetEntity="Folder", mappedBy="parent")
	 * @ORM\OrderBy({"lft" = "ASC"})
	 */
	protected Collection $children;

	/**
	 * @var Collection<File>
	 * @ORM\OneToMany(targetEntity="File", mappedBy="folder")
	 */
	protected Collection $files;

	public function __construct($title)
	{
		$this->title    = $title;
		$this->children = new ArrayCollection();
		$this->files    = new ArrayCollection();
		$this->status   = 1;
	}

	public function getParent(): ?Folder { return $this->parent; }

	public function setParent(?Folder $parent = null): void { $this->parent = $parent; }

	public function getChildrens(): Collection { return $this->children; }

	public function getLvl(): ?int { return $this->lvl; }

	public function getFiles(): Collection { return $this->files; }

	/**
	 * @return File[]
	 */
	public function getPublishedFiles(): array
	{
		$cr = Criteria::create();

		return $this->files->matching($cr->where(Criteria::expr()->eq('status', self::STATUS_PUBLISHED)))->toArray();
	}
}
