<?php declare(strict_types = 1);

namespace Mall\AdminModule\Model\Subscribers;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\ControlEvent;
use Core\Model\Event\Event;
use Core\Model\Event\GridFilterEvent;
use Doctrine\Common\Collections\Criteria;
use EshopOrders\AdminModule\Components\Order\OrderForm;
use EshopOrders\AdminModule\Components\Order\OrdersGrid;
use EshopOrders\Model\Entities\Order;
use Mall\Model\OrdersExported;
use Nette\Localization\Translator;
use Nette\Utils\Html;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrderSubscriber implements EventSubscriberInterface
{
	/** @var array|null */
	protected static ?array          $allOrders = null;
	protected OrdersExported         $ordersExported;
	protected EntityManagerDecorator $em;
	protected Translator             $translator;

	public function __construct(
		OrdersExported         $ordersExported,
		EntityManagerDecorator $em,
		Translator             $translator
	)
	{
		$this->ordersExported = $ordersExported;
		$this->em             = $em;
		$this->translator     = $translator;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			OrderForm::class . '::onAttach'               => 'orderFormAttached',
			OrdersGrid::class . '::columnSpeditionRender' => 'columnSpeditionRender',
			OrdersGrid::class . '::columnIdRender'        => 'columnIdRender',
			OrdersGrid::class . '::filterPackageNumber'   => 'gridFilterPackageNumber',
			OrdersGrid::class . '::filterId'              => 'gridFilterId',
		];
	}

	public function orderFormAttached(ControlEvent $event): void
	{
		/** @var OrderForm $control */
		$control   = $event->control;
		$order     = $control->order;
		$spedition = $order->getSpedition() ? $order->getSpedition()->getSpedition() : null;

		if ($spedition && $spedition->getIdent() === 'mdp') {
			$control->template->mallExport                  = $this->ordersExported->getOrders([$order->getId()])[$order->getId()] ?? null;
			$control->template->speditionExportInfoTemplate = __DIR__ . '/exportInfoTemplate.latte';
		}
	}

	public function columnSpeditionRender(Event $event): void
	{
		$data           = $event->data;
		$speditionIdent = 'mdp';
		$export         = null;

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		if ($data['ordersBySpedition'][$speditionIdent] ?? null) {
			$export = $this->getOrders(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []))[$order->getId()] ?? null;
		} else if ($order->getSpeditionIdent() === $speditionIdent) {
			$export = $this->ordersExported->getOrders([$order->getId()]);
		}

		if ($export && $export['exported'] !== null) {
			$html->addHtml(Html::el('div')
				->addHtml(Html::el('a', ['target' => '_blank'])
					->setAttribute('href', $export['trackingUrl'])
					->setText($export['trackingNumber'])
				));
		}
	}

	public function columnIdRender(Event $event): void
	{
		$data = $event->data;

		/** @var Order $order */
		$order = $data['order'];

		/** @var Html $html */
		$html = $data['html'];

		foreach (['mall', 'mdp'] as $speditionIdent) {
			if ($data['ordersBySpedition'][$speditionIdent]) {
				$export = $this->getOrders(array_keys($data['ordersBySpedition'][$speditionIdent] ?? []))[$order->getId()] ?? null;
				continue;
			} else if ($order->getSpeditionIdent() === $speditionIdent) {
				$export = $this->ordersExported->getOrders([$order->getId()]);
				continue;
			} else {
				$export = null;
			}
		}

		if ($export) {
			$html->addHtml(Html::el('div')
				->setText($export['mallId']));
		}
	}

	public function gridFilterPackageNumber(GridFilterEvent $event): void
	{
		$ids = $this->ordersExported->findIdByPackageNumber($event->value);

		if ($ids) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	public function gridFilterId(GridFilterEvent $event): void
	{
		$ids = $this->ordersExported->findIdByMallId((int) $event->value);

		if ($ids) {
			$event->criteria->orWhere(Criteria::expr()->in('o.id', $ids));
		}
	}

	/**
	 * @param string[]|int[] $ids
	 */
	protected function getOrders(array $ids): ?array
	{
		if (self::$allOrders === null) {
			self::$allOrders = $this->ordersExported->getOrders($ids);
		}

		return self::$allOrders;
	}
}
