<?php declare(strict_types = 1);

namespace Mall\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\DBAL\Exception;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\OrderStatus;
use Mall\Model\Entities\MallOrder;

class OrdersExported
{
	protected EntityManagerDecorator $em;
	protected MallClients            $mallClients;

	public function __construct(
		EntityManagerDecorator $em,
		MallClients            $mallClients
	)
	{
		$this->em          = $em;
		$this->mallClients = $mallClients;
	}

	public function getOrders(array $ids): array
	{
		$data = [];

		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(mo.order) as order, mo.mallId , mo.trackingNumber, mo.trackingUrl, mo.exported')
			         ->from(MallOrder::class, 'mo')
			         ->where('mo.order IN (:ids)')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getArrayResult() as $row) {
			$data[$row['order']] = $row;
		}

		return $data;
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];
		foreach ($this->em->getRepository(MallOrder::class)->createQueryBuilder('mo')
			         ->select('IDENTITY(mo.order) as id')
			         ->where('mo.trackingNumber LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}

	public function findIdByMallId(int $mallId): array
	{
		$ids = [];
		foreach ($this->em->getRepository(MallOrder::class)->createQueryBuilder('mo')
			         ->select('IDENTITY(mo.order) as id')
			         ->where('mo.mallId LIKE :mallId')
			         ->setParameter('mallId', "%$mallId%")
			         ->getQuery()->getArrayResult() as $row) {
			$ids[] = $row['id'];
		}

		return $ids;
	}

	/**
	 * @return MallOrder[]
	 */
	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ($this->em->createQueryBuilder()->select('mo')
			         ->from(MallOrder::class, 'mo')
			         ->where('mo.lastStatus NOT IN (:status)')
			         ->setParameters([
				         'status' => [
					         MallOrder::STATUS_DELIVERED,
					         MallOrder::STATUS_RETURNED,
					         MallOrder::STATUS_CANCELED,
					         MallOrder::STATUS_LOST,
					         MallOrder::STATUS_STORNO,
				         ],
			         ])->getQuery()->getResult() as $row) {
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	public function setLastStatus(array $orderIds, string $status = MallOrder::STATUS_DELIVERED): void
	{
		$this->em->getConnection()->executeStatement("UPDATE mall__order SET last_status = ? WHERE order_id IN (" . implode(',', $orderIds) . ")", [
			$status,
		]);
	}

}
