<?php declare(strict_types = 1);

namespace Mall\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\Query\Parameter;
use Mall\Model\Entities\MallOrder;

class OrdersExported
{
	public function __construct(
		protected EntityManagerDecorator $em,
	)
	{
	}

	public function getOrders(array $ids): array
	{
		$data = [];

		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(mo.order) as order, mo.mallId , mo.trackingNumber, mo.trackingUrl, mo.exported')
			         ->from(MallOrder::class, 'mo')
			         ->where('mo.order IN (:ids)')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getArrayResult() as $row) {
			/** @var array $row */
			$data[$row['order']] = $row;
		}

		return $data;
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];
		foreach ($this->em->getRepository(MallOrder::class)->createQueryBuilder('mo')
			         ->select('IDENTITY(mo.order) as id')
			         ->where('mo.trackingNumber LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row) {
			/** @var array $row */
			$ids[] = $row['id'];
		}

		return $ids;
	}

	public function findIdByMallId(int $mallId): array
	{
		$ids = [];
		foreach ($this->em->getRepository(MallOrder::class)->createQueryBuilder('mo')
			         ->select('IDENTITY(mo.order) as id')
			         ->where('mo.mallId LIKE :mallId')
			         ->setParameter('mallId', "%$mallId%")
			         ->getQuery()->getArrayResult() as $row) {
			/** @var array $row */
			$ids[] = $row['id'];
		}

		return $ids;
	}

	/**
	 * @return MallOrder[]
	 */
	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ((array) $this->em->createQueryBuilder()->select('mo')
			->from(MallOrder::class, 'mo')
			->where('mo.lastStatus NOT IN (:status)')
			->setParameters(new ArrayCollection([new Parameter('status', [
				MallOrder::STATUS_DELIVERED,
				MallOrder::STATUS_RETURNED,
				MallOrder::STATUS_CANCELED,
				MallOrder::STATUS_LOST,
				MallOrder::STATUS_STORNO,
			])]))->getQuery()->getResult() as $row) {
			/** @var MallOrder $row */
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	public function setLastStatus(array $orderIds, string $status = MallOrder::STATUS_DELIVERED): void
	{
		$this->em->getConnection()->executeStatement("UPDATE mall__order SET last_status = ? WHERE order_id IN (" . implode(',', $orderIds) . ")", [
			$status,
		]);
	}

}
