<?php declare(strict_types = 1);

namespace Mall\AdminModule\Components\Categories;

use Core\Model\UI\BaseControl;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\FrontModule\Model\Features;
use Mall\Model\Entities\CategoryMap;
use Mall\Model\Entities\CategoryParam;
use Mall\Model\Entities\Param;
use Mall\Model\Services\CategoriesService;
use EshopCatalog\FrontModule\Model\Categories as EshopFrontCategories;
use Mall\Model\Services\ParametersService;

class CategoriesGrid extends BaseControl
{
	protected string $country;

	protected CategoriesService $categoriesService;

	protected EshopFrontCategories $eshopFrontCategories;

	protected ParametersService $parametersService;

	protected Features $featuresService;

	public function __construct(string $country, CategoriesService $categoriesService, EshopFrontCategories $eshopFrontCategories,
	                            ParametersService $parametersService, Features $features)
	{
		$this->country              = $country;
		$this->categoriesService    = $categoriesService;
		$this->eshopFrontCategories = $eshopFrontCategories;
		$this->parametersService    = $parametersService;
		$this->featuresService      = $features;
	}

	public function render(): void
	{
		$this->template->autocompleteUrl = $this->getPresenter()->link(":EshopCatalog:Cron:Categories:categories");
		$this->template->setFeatureLink  = $this->link('setFeature!');
		$this->template->addLink         = $this->link('add!');
		$this->template->removeLink      = $this->link('remove!');
		$this->template->categories      = $this->getCategories();
		$this->template->features        = $this->getFeatures();
		$this->template->country         = $this->country;
		$this->template->render($this->getTemplateFile());
	}

	/*******************************************************************************************************************
	 * ===============  handle
	 */

	public function handleSetFeature(): void
	{
		$paramId = $this->getPost('paramId');
		$feature = $this->getPost('feature');

		$result = $this->parametersService->setFeature($paramId, $feature);
		if ($result)
			$this->getPresenter()->flashMessageSuccess('default.saved');
		else
			$this->getPresenter()->flashMessageDanger('default.error');
		$this->getPresenter()->payload->result = $result ? 'ok' : 'error';
		$this->getPresenter()->redrawControl('flashes');
	}

	public function handleAdd(): void
	{
		$catId  = $this->getPost('id');
		$shopId = $this->getPost('shopId');

		$result = $this->categoriesService->addToMap($catId, $shopId);
		if ($result)
			$this->getPresenter()->flashMessageSuccess('default.saved');
		else
			$this->getPresenter()->flashMessageDanger('default.error');
		$this->getPresenter()->payload->result = $result ? 'ok' : 'error';
		$this->getPresenter()->redrawControl('flashes');
	}

	public function handleRemove(): void
	{
		$catId  = $this->getPost('id');
		$shopId = $this->getPost('shopId');

		$result = $this->categoriesService->removeFromMap($catId, $shopId);
		if ($result)
			$this->getPresenter()->flashMessageSuccess('default.removed');
		else
			$this->getPresenter()->flashMessageDanger('default.removeFailed');
		$this->getPresenter()->payload->result = $result ? 'ok' : 'error';
		$this->getPresenter()->redrawControl('flashes');
	}

	/*******************************************************************************************************************
	 * ===============  Getters
	 */

	protected function getCategories(): array
	{
		$arr   = [];
		$roots = $this->eshopFrontCategories->getAllRootIds();
		$roots = array_flip($roots);

		foreach ($this->categoriesService->getEr()->createQueryBuilder('c')->select('c.id, c.mallId, c.title')
			         ->andWhere('c.country = :country')
			         ->andWhere('c.isAvailable = 1')
			         ->setParameters([
				         'country' => $this->country,
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			$arr[$row['id']] = [
				'mallId'         => $row['mallId'],
				'title'          => $row['title'],
				'shopCategories' => [],
				'params'         => [],
			];
		}

		foreach ($this->em->getRepository(CategoryMap::class)->createQueryBuilder('cm')
			         ->select('catM.id as id, IDENTITY(cm.category) as shopId, catT.name as catName')
			         ->innerJoin('cm.mallCategory', 'catM', Join::WITH, 'catM.country = :country')
			         ->innerJoin('cm.category', 'cat')
			         ->innerJoin('cat.categoryTexts', 'catT', Join::WITH, 'catT.lang = :locale OR catT.lang = :lang')
			         ->setParameters([
				         'lang'    => $this->country,
				         'country' => $this->country,
				         'locale'  => $this->translator->getLocale(),
			         ])
			         ->getQuery()->getArrayResult() as $row) {
			if (!isset($arr[$row['id']]))
				continue;

			$daoCat = $this->eshopFrontCategories->findInAll($row['shopId']);

			if ($daoCat) {
				$path                                              = $daoCat->getParentPathString();
				$arr[$row['id']]['shopCategories'][$row['shopId']] = $roots[$daoCat->getRootId()] . ' > ' . ($path ? ($path . ' > ') : '') . $daoCat->name;
			}
		}

		foreach ($this->em->getRepository(Param::class)->createQueryBuilder('p')
			         ->select('IDENTITY(p.category) as catId, p.id as paramId, p.mallId, p.title, p.feature')
			         ->getQuery()->getArrayResult() as $row) {
			if (isset($arr[$row['catId']]))
				$arr[$row['catId']]['params'][$row['paramId']] = $row;
		}

		return $arr;
	}

	protected function getFeatures(): array
	{
		$arr = [
			Param::FEATURE_SUBCATEGORIES => [
				'id'    => Param::FEATURE_SUBCATEGORIES,
				'title' => $this->translator->translate('mall.param.subcategories'),
			],
		];

		foreach ($this->featuresService->getPublishedFeatures() as $k => $v)
			$arr[$k] = [
				'id'    => $k,
				'title' => $v,
			];

		return $arr;
	}

	/*******************************************************************************************************************
	 * ===============  COMPONENTS
	 */
}
