<?php declare(strict_types = 1);

namespace Mall\AdminModule\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Helpers\BaseService;
use DateTimeInterface;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderStatus;
use Mall\Model\Entities\MallOrder;
use Mall\Model\MallClients;

class AccountantService extends BaseService
{
	protected EntityManagerDecorator $em;

	protected MallClients $mallClients;

	public function __construct(EntityManagerDecorator $em, MallClients $mallClients)
	{
		$this->em          = $em;
		$this->mallClients = $mallClients;
	}

	protected function getOrders(string $country, DateTimeInterface $from, DateTimeInterface $to): array
	{
		$data = [];

		foreach ($this->em->createQueryBuilder()->select('IDENTITY(mo.order) as id, mo.mallId')
			         ->from(MallOrder::class, 'mo')
			         ->innerJoin(OrderStatus::class, 'os', Join::WITH, 'os.order = mo.order AND os.status = :oStatus AND os.created >= :from AND os.created <= :to')
			         ->where('mo.country = :country')
			         ->andWhere('mo.lastStatus = :lastStatus')
			         ->setParameters([
				         'country'    => $country,
				         'oStatus'    => OrderStatus::STATUS_FINISHED,
				         'from'       => $from,
				         'to'         => $to,
				         'lastStatus' => MallOrder::STATUS_DELIVERED,
			         ])->getQuery()->getArrayResult() as $row) {
			$data[$row['id']] = $row['mallId'];
		}

		return $data;
	}

	public function exportOverview(string $country, DateTimeInterface $from, DateTimeInterface $to): array
	{
		$data   = [];
		$orders = $this->getOrders($country, $from, $to);
		$client = $this->mallClients->getClient($country);

		foreach ($this->em->createQueryBuilder()
			         ->select('o, cus, op, os, del, inv, cur')
			         ->from(Order::class, 'o')
			         ->leftJoin('o.customer', 'cus')
			         ->leftJoin('o.payment', 'op')
			         ->leftJoin('o.spedition', 'os')
			         ->leftJoin('o.addressDelivery', 'del')
			         ->leftJoin('o.addressInvoice', 'inv')
			         ->leftJoin('o.currency', 'cur')
			         ->andWhere('o.id IN (:ids)')
			         ->setParameters([
				         'ids' => array_keys($orders),
			         ])
			         ->groupBy('o.id')
			         ->getQuery()->getResult() as $row) {
			/** @var Order $row */
			$data[$orders[$row->getId()]] = [
				'mallId'   => $orders[$row->getId()],
				'price'    => str_replace('.', ',', (string) $row->getPrice(true)),
				'currency' => $row->getCurrencyCode(),
				'vs'       => $orders[$row->getId()],
				'customer' => $client->invoiceId,
			];
		}

		return $data;
	}

	public function exportControlList(string $country, DateTimeInterface $from, DateTimeInterface $to): array
	{
		$data           = [];
		$mallOrders     = $this->getOrders($country, $from, $to);
		$client         = $this->mallClients->getClient($country);
		$ordersStatuses = [];

		foreach ($this->em->createQueryBuilder()
			         ->select('os.id, IDENTITY(os.order) as orderId, IDENTITY(os.status) as status, os.created, os.deleted')
			         ->from(OrderStatus::class, 'os')
			         ->where('os.order IN (:ids)')
			         ->andWhere('os.deleted IS NULL OR os.deleted = 0')
			         ->setParameters([
				         'ids' => array_keys($mallOrders),
			         ])->getQuery()->getArrayResult() as $row) {
			$ordersStatuses[$row['orderId']][$row['status']] = $row['created'];
		}

		foreach ($this->em->createQueryBuilder()
			         ->select('oi, oit, o, cus, op, os, del, inv, cur')
			         ->from(OrderItem::class, 'oi')
			         ->innerJoin('oi.order', 'o')
			         ->innerJoin('oi.orderItemTexts', 'oit')
			         ->leftJoin('o.customer', 'cus')
			         ->leftJoin('o.payment', 'op')
			         ->leftJoin('o.spedition', 'os')
			         ->leftJoin('o.addressDelivery', 'del')
			         ->leftJoin('o.addressInvoice', 'inv')
			         ->leftJoin('o.currency', 'cur')
			         ->andWhere('o.id IN (:ids)')
			         ->setParameters([
				         'ids' => array_keys($mallOrders),
			         ])
			         ->groupBy('oi.id')
			         ->getQuery()->getResult() as $row) {
			/** @var OrderItem $row */
			$order           = $row->getOrder();
			$priceWithoutVat = $row->getPriceWithoutVat(true);
			$commission      = (int) $row->getMoreDataValue('commission');
			$commissionPrice = round(($commission / 100) * $priceWithoutVat, 2);
			$buyPrice        = $priceWithoutVat - $commissionPrice;
			$created         = $ordersStatuses[$order->getId()][OrderStatus::STATUS_CREATED] ?? null;
			$delivered       = $ordersStatuses[$order->getId()][OrderStatus::STATUS_FINISHED] ?? null;
			if (!$created || !$delivered)
				continue;

			$data[$row->getId()] = [
				'mallId'     => $mallOrders[$order->getId()],
				'customer'   => $order->getAddressInvoice()->getName(),
				'item'       => $row->getOrderItemText()->getName(),
				'created'    => $created->format('j.n.Y H:i'),
				'delivered'  => $delivered->format('j.n.Y H:i'),
				'vat'        => $row->getVatRate(),
				'price'      => str_replace('.', ',', (string) $priceWithoutVat),
				'commission' => str_replace('.', ',', (string) $commissionPrice),
				'buyPrice'   => str_replace('.', ',', (string) $buyPrice),
			];
		}

		return $data;
	}
}
