<?php declare(strict_types = 1);

namespace Mall\AdminModule\Model\Subscribers;

use Core\Model\Event\CreateFormEvent;
use Core\Model\Event\FormSuccessEvent;
use Core\Model\Event\SetFormDataEvent;
use Core\Model\Helpers\Strings;
use Core\Model\UI\Form\BaseContainer;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\AdminModule\Components\Products\ProductForm;
use EshopCatalog\Model\Entities\Product;
use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\Model\Entities\ProductInSite;
use Mall\Model\Entities\CategoryMap;
use Mall\Model\Entities\ProductLabel;
use Mall\Model\Entities\ProductMap;
use Mall\Model\Entities\ProductPromotion;
use Mall\Model\MallClients;
use Mall\Model\Services\LabelsService;
use Mall\Model\Services\ProductLabelsService;
use Mall\Model\Services\ProductMapService;
use Mall\Model\Services\ProductPromotionsService;
use Mall\Model\Subscribers\ProductSubscriber;
use Mall\Model\Sync\SyncProduct;
use Nette\Http\Request;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Mall\Model\Entities\Category as MallCategory;

class ProductFormSubscriber implements EventSubscriberInterface
{
	protected EntityManagerDecorator $em;

	protected MallClients $mallClients;

	protected LabelsService $labelsService;

	protected ProductLabelsService $productLabelsService;

	protected ProductPromotionsService $productPromotionsService;

	protected ProductMapService $productMapService;

	protected Request $httpRequest;

	protected SyncProduct $syncProduct;

	protected static ?array $isInMall = null;

	public function __construct(EntityManagerDecorator $em, MallClients $mallClients, LabelsService $labelsService,
	                            ProductLabelsService   $productLabelsService, ProductPromotionsService $productPromotionsService,
	                            Request                $httpRequest, ProductMapService $productMapService, SyncProduct $syncProduct)
	{
		$this->em                       = $em;
		$this->mallClients              = $mallClients;
		$this->labelsService            = $labelsService;
		$this->productLabelsService     = $productLabelsService;
		$this->productPromotionsService = $productPromotionsService;
		$this->httpRequest              = $httpRequest;
		$this->productMapService        = $productMapService;
		$this->syncProduct              = $syncProduct;
	}

	public static function getSubscribedEvents(): array
	{
		return [
			ProductForm::class . '::createForm'           => 'createForm',
			ProductForm::class . '::formSuccessAfterSave' => ['formSuccessAfterSave', -200],
			ProductForm::class . '::setProduct'           => 'setProduct',
		];
	}

	protected function checkMall()
	{
		if (self::$isInMall === null) {
			self::$isInMall = [];
			$urlPathArray   = explode('/', $this->httpRequest->getUrl()->getPath());
			$id             = (int) end($urlPathArray);

			$baseCategories = [];
			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(p.category) as cat')
				         ->from(ProductInSite::class, 'p')
				         ->where('p.isActive = 1')
				         ->andWhere('p.product = :prod')
				         ->setParameter('prod', $id)
				         ->getQuery()->getScalarResult() as $row)
				$baseCategories[] = $row['cat'];

			foreach ($this->em->createQueryBuilder()
				         ->select('IDENTITY(cm.mallCategory) as cat, mallC.country')
				         ->from(CategoryMap::class, 'cm')
				         ->innerJoin('cm.mallCategory', 'mallC', Join::WITH, 'mallC.isAvailable = 1')
				         ->where('cm.category IN (:cats)')
				         ->setParameter('cats', $baseCategories)
				         ->getQuery()->getArrayResult() as $row) {
				if (!isset(self::$isInMall[$row['country']]))
					self::$isInMall[$row['country']] = [];

				self::$isInMall[$row['country']][] = $row['cat'];
			}
		}

		return self::$isInMall;
	}

	public function createForm(CreateFormEvent $event): void
	{
		$this->checkMall();

		if (empty(self::$isInMall))
			return;

		$containers  = $event->form->addContainer('mall');
		$labels      = $this->labelsService->getVisibleArray();
		$mallClients = $this->mallClients->getClients();

		$containers->addCustomData('mallClients', $mallClients);

		foreach ($mallClients as $country => $client) {
			if (!isset(self::$isInMall[$country]))
				continue;

			$container = $containers->addContainer($country);

			$container->addBool('active', 'default.isActive')
				->setDefaultValue(1);

			$labelsContainer = $container->addContainer('labels');
			foreach ($labels[$country] ?? [] as $mallId => $v) {
				$container2 = new BaseContainer();
				$container2->addBool('active', $mallId . ' - ' . $v['title']);
				$container2->addDateTimePicker('dateFrom', 'mall.label.from');
				$container2->addDateTimePicker('dateTo', 'mall.label.to');
				$container2->addHidden('id', (string) $v['id']);

				$labelsContainer->addComponent($container2, $mallId);
			}

			$promoContainer = $container->addContainer('promotions');
			$container2     = new BaseContainer();
			$container2->addText('price', 'mall.promotion.price')
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0)
				->setRequired(false);
			$container2->addDateTimePicker('dateFrom', 'mall.promotion.from');
			$container2->addDateTimePicker('dateTo', 'mall.promotion.to');
			$promoContainer->addComponent($container2, '0');

			$container2 = $container->addContainer('general', 'mall.generalForm.title');
			$container2->addText('price', 'mall.generalForm.price')
				->setHtmlType('number')
				->setHtmlAttribute('step', .01)
				->setHtmlAttribute('min', 0);
		}

		$event->form->getRenderer()->extendedLayout['mall.title.productFormTab'] = __DIR__ . '/ProductFormMall.latte';
	}

	public function formSuccessAfterSave(FormSuccessEvent $event): void
	{
		$this->checkMall();
		if (empty(self::$isInMall))
			return;

		/** @var Product $product */
		$product = $event->custom['entity'];
		$values  = $event->values;

		if (!$values->mall)
			return;

		$currentLabels     = $this->productLabelsService->getForProduct($product->getId());
		$currentPromotions = $this->productPromotionsService->getForProduct($product->getId());
		$productMaps       = $this->productMapService->getByEshopProduct($product->getId(), true);

		foreach ($this->mallClients->getClients() as $country => $client) {
			if (!isset(self::$isInMall[$country]))
				continue;

			$productMap = $productMaps[$country] ?? null;

			if (!$productMap) {
				$productMap = new ProductMap($product, $country);
			}

			$generalValues     = $values['mall'][$country]['general'];
			$productMap->price = Strings::formatEntityDecimal($generalValues['price']);

			$productMap->isActive = (int) $values['mall'][$country]['active'];
			$this->em->persist($productMap);
			$this->em->flush($productMap);

			$formLabels = [];
			foreach ($values['mall'][$country]['labels'] ?? [] as $mallId => $vals) {
				if ($vals->active && $vals->dateFrom && $vals->dateTo)
					$formLabels[$mallId] = $vals;
			}

			$formPromotions = [];
			foreach ($values['mall'][$country]['promotions'] ?? [] as $k => $vals) {
				if ($vals->price && $vals->dateFrom && $vals->dateTo) {
					$formPromotions[$k] = $vals;
				}
			}

			if (!isset($currentLabels[$country]))
				$currentLabels[$country] = [];
			if (!isset($currentPromotions[$country]))
				$currentPromotions[$country] = [];

			// Nove labels
			foreach (array_diff_key($formLabels, $currentLabels[$country]) as $mallId => $vals) {
				$entity = new ProductLabel($product, $this->labelsService->getReference($vals['id']), $vals->dateFrom, $vals->dateTo);
				$this->em->persist($entity);
			}

			// Stare labels
			foreach (array_diff_key($currentLabels[$country], $formLabels) as $mallId => $data) {
				$entity = $this->productLabelsService->getReference($data['id']);
				$this->em->remove($entity);
			}

			foreach (array_diff_key($formPromotions, $currentPromotions[$country]) as $vals) {
				$entity = new ProductPromotion($product, $country, Strings::formatEntityDecimal($vals->price), $vals->dateFrom, $vals->dateTo);
				$this->em->persist($entity);
			}

			foreach (array_diff_key($currentPromotions[$country], $formPromotions) as $data) {
				$entity = $this->productPromotionsService->getReference($data['id']);
				$this->em->remove($entity);
			};

			$this->em->flush();
		}

		$this->productMapService->clearClassCache();

		if (!$product->isPublished)
			$this->syncProduct->sendInactive($product->getId());
		else
			ProductSubscriber::$updateFull[] = $product->getId();
	}

	public function setProduct(SetFormDataEvent $event)
	{
		$this->checkMall();
		if (empty(self::$isInMall))
			return;

		/** @var Product $product */
		$product = $event->entity;

		$d = [];

		foreach ($this->productLabelsService->getForProduct($product->getId()) as $country => $rows) {
			foreach ($rows as $mallId => $vals) {
				$d[$country]['labels'][$mallId] = [
					'active'   => 1,
					'dateFrom' => $vals['from'],
					'dateTo'   => $vals['to'],
					'id'       => $vals['id'],
				];
			}
		}

		foreach ($this->productPromotionsService->getForProduct($product->getId()) as $country => $rows) {
			foreach ($rows as $id => $vals) {
				$d[$country]['promotions'][] = [
					'price'    => $vals['price'],
					'dateFrom' => $vals['from'],
					'dateTo'   => $vals['to'],
				];
			}
		}

		$event->form->onAnchor[] = function($form) use ($product) {
			$form->getParent()->template->mallProductExist = $this->productMapService->getByEshopProduct($product->getId());
		};

		foreach ($this->productMapService->getByEshopProduct($product->getId(), true) as $country => $productMap) {
			$d[$country]['active']           = (int) $productMap->isActive;
			$d[$country]['general']['price'] = $productMap->price;
		}

		$event->form['mall']->setDefaults($d);
	}
}
