<?php declare(strict_types = 1);

namespace Mall\Model\Dao\Product;

use Mall\Model\MallConfig;
use Mall\Model\ProductHelper;

class Product extends BaseProduct
{
	const STAGE_LIVE  = 'live';
	const STAGE_DRAFT = 'draft';

	const PACKAGE_SIZE_SMALLBOX = 'smallbox';
	const PACKAGE_SIZE_BIGBOX   = 'bigbox';

	public string $category_id;

	public string $stage = self::STAGE_DRAFT;

	public ?string $brand_id = null;

	/** @var Variant[] */
	public array $variants = [];

	public array $variable_parameters = [];

	public int $vat = 21;

	public $eshopId;

	public string $id;

	public string $title;

	public ?int $barcode = null;

	public int $priority = 1;

	public float $price;

	public ?float $rrp = null;

	protected array $parameters = [];

	/** @var Media[] */
	protected array $media = [];

	/** @var Promotion[] */
	protected array $promotions = [];

	/** @var Label[] */
	protected array $labels = [];

	public Availability $availability;

	protected array $recommended = [];

	public int $delivery_delay = 1;

	public bool $free_delivery = false;

	protected string $package_size = Product::PACKAGE_SIZE_SMALLBOX;

	public bool $mallbox_allowed = false;

	public ?string $partner_title = null;

	public function __construct(string $category_id, string $id, string $title, string $shortdesc, string $longdesc, float $price, Availability $availability)
	{
		$this->category_id  = $category_id;
		$this->id           = $id;
		$this->title        = $title;
		$this->longdesc     = $longdesc;
		$this->price        = $price;
		$this->availability = $availability;
		$this->setShortDesc($shortdesc);
	}

	public function addVariant(Variant $variant): self
	{
		$this->variants[] = $variant;

		return $this;
	}

	public function addParameter(string $key, string $value): self
	{
		$forReplace = MallConfig::load('paramValueReplaces')[$value] ?? null;
		if ($forReplace)
			$value = $forReplace;

		$this->parameters[$key] = $value;

		return $this;
	}

	public function addVariableParameter(string $key): self
	{
		if (in_array($key, $this->variable_parameters))
			return $this;

		$this->variable_parameters[] = $key;

		return $this;
	}

	public function addMedia(Media $media): self
	{
		$this->media[] = $media;

		return $this;
	}

	public function addPromotion(Promotion $promotion): self
	{
		$this->promotions[] = $promotion;

		return $this;
	}

	public function addLabel(Label $label): self
	{
		$this->labels[] = $label;

		return $this;
	}

	public function setPackageSize(string $packageSize): self
	{
		if (!ProductHelper::checkPackageSize($packageSize))
			throw new \Exception("Package size '$packageSize' not found");

		$this->package_size = $packageSize;

		return $this;
	}

	public function addRecommended(string $productId): self
	{
		$this->recommended[] = $productId;

		return $this;
	}
}
