<?php declare(strict_types = 1);

namespace Mall\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\OrderStatus;
use Mall\Model\Entities\MallOrder;
use Mall\Model\Services\MallSellersService;

class OrdersExported
{
	protected EntityManagerDecorator $em;

	protected MallSellersService $mallSellersService;

	protected MallClients $mallClients;

	public function __construct(EntityManagerDecorator $em, MallSellersService $mallSellersService, MallClients $mallClients)
	{
		$this->em                 = $em;
		$this->mallSellersService = $mallSellersService;
		$this->mallClients        = $mallClients;
	}

	/**
	 * @param array $ids
	 *
	 * @return MallOrder[]
	 */
	public function getOrdersNotExported(array $ids = []): array
	{
		$orders = [];
		foreach ($this->em->getRepository(MallOrder::class)->createQueryBuilder('mo')
			         ->addSelect('o, os')
			         ->innerJoin('mo.order', 'o', Join::WITH, 'o.id IN (:ids)')
			         ->innerJoin('o.spedition', 'os')
			         ->where('mo.exported IS NULL')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getResult() as $mo) {
			/** @var MallOrder $mo */
			if ($mo->country) {
				$mo->mallClient = $this->mallClients->getClient($mo->country);
			} else {
				// Pokud neni vyplnena zeme tak najit a doplnit
				foreach ($this->mallSellersService->getExisting() as $seller) {
					/** @var MallOrder $mo */
					if (in_array($mo->getOrder()->site->getIdent(), $seller->seller->getSitesIdent())) {
						$mo->mallClient = $this->mallClients->getClient($seller->country);
						$this->em->getConnection()->executeStatement("UPDATE mall__order SET country = ? WHERE order_id = ?", [
							$mo->mallClient->getCountry(),
							$mo->getOrder()->getId(),
						]);

						break;
					}
				}
			}

			if (!$mo->mallClient)
				continue;

			$orders[$mo->getOrder()->getId()] = $mo;
		}

		$used = [];
		foreach ($this->em->createQueryBuilder()->select('IDENTITY(os.order) as order, IDENTITY(os.status) as status')
			         ->from(OrderStatus::class, 'os')
			         ->where('os.order IN (:ids)')
			         ->setParameters([
				         'ids' => array_keys($orders),
			         ])->orderBy('os.created', 'DESC')
			         ->getQuery()->getArrayResult() as $row) {
			if (in_array($row['order'], $used))
				continue;
			$used[] = $row['order'];

			if ($row['status'] !== OrderStatus::STATUS_PROCESSING)
				unset($orders[$row['order']]);
		}

		return $orders;
	}

	public function getOrdersExported(array $ids): array
	{
		$data = [];

		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(mo.order) as order, mo.mallId , mo.trackingNumber, mo.trackingUrl, mo.exported')
			         ->from(MallOrder::class, 'mo')
			         ->where('mo.order IN (:ids)')
			         ->andWhere('mo.exported IS NOT NULL')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getArrayResult() as $row)
			$data[$row['order']] = $row;

		return $data;
	}

	public function getOrders(array $ids): array
	{
		$data = [];

		foreach ($this->em->createQueryBuilder()
			         ->select('IDENTITY(mo.order) as order, mo.mallId , mo.trackingNumber, mo.trackingUrl, mo.exported')
			         ->from(MallOrder::class, 'mo')
			         ->where('mo.order IN (:ids)')
			         ->setParameter('ids', $ids)
			         ->getQuery()->getArrayResult() as $row)
			$data[$row['order']] = $row;

		return $data;
	}

	public function findIdByPackageNumber(string $packageNumber): array
	{
		$ids = [];
		foreach ($this->em->getRepository(MallOrder::class)->createQueryBuilder('mo')
			         ->select('IDENTITY(mo.order) as id')
			         ->where('mo.trackingNumber LIKE :number')
			         ->setParameter('number', "%$packageNumber%")
			         ->getQuery()->getArrayResult() as $row)
			$ids[] = $row['id'];

		return $ids;
	}

	public function findIdByMallId(int $mallId): array
	{
		$ids = [];
		foreach ($this->em->getRepository(MallOrder::class)->createQueryBuilder('mo')
			         ->select('IDENTITY(mo.order) as id')
			         ->where('mo.mallId = :mallId')
			         ->setParameter('mallId', $mallId)
			         ->getQuery()->getArrayResult() as $row)
			$ids[] = $row['id'];

		return $ids;
	}

	/**
	 * @return MallOrder[]
	 */
	public function getOrdersNotCompleted(): array
	{
		$data = [];

		foreach ($this->em->createQueryBuilder()->select('mo')
			         ->from(MallOrder::class, 'mo')
			         ->where('mo.lastStatus NOT IN (:status)')
			         ->setParameters([
				         'status' => [
					         MallOrder::STATUS_DELIVERED,
					         MallOrder::STATUS_RETURNED,
					         MallOrder::STATUS_CANCELED,
					         MallOrder::STATUS_LOST,
					         MallOrder::STATUS_STORNO,
				         ],
			         ])->getQuery()->getResult() as $row) {
			$data[$row->getOrder()->getId()] = $row;
		}

		return $data;
	}

	public function setLastStatus(array $orderIds, string $status = MallOrder::STATUS_DELIVERED): void
	{
		$this->em->getConnection()->executeStatement("UPDATE mall__order SET last_status = ? WHERE order_id IN (" . implode(',', $orderIds) . ")", [
			$status,
		]);
	}
}
