<?php declare(strict_types = 1);

namespace Mall\Model;

use Core\Model\Event\DaoEvent;
use Core\Model\Event\EventDispatcher;
use EshopCatalog\Model\Entities\CategoryTexts;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Sites;
use EshopCatalog\FrontModule\Model\ProductsFacade;
use EshopCatalog\Model\Entities\Availability;
use EshopCatalog\Model\Entities\Category;
use Mall\Model\Dao\MallClient;
use Mall\Model\Entities\Param;
use Mall\Model\Services\MallApiBrands;
use Mall\Model\Services\ParametersService;
use Mall\Model\Dao\Product as Dao;
use EshopCatalog\FrontModule\Model\Dao as EshopDao;
use Mall\Model\Services\ProductLabelsService;
use Mall\Model\Services\ProductMapService;
use Mall\Model\Services\ProductPromotionsService;

class ProductBuilder
{
	protected ProductsFacade $productsFacade;

	protected ParametersService $parametersService;

	protected ProductMapService $productMapService;

	protected ProductPromotionsService $productPromotionsService;

	protected ProductLabelsService $productLabelsService;

	protected MallApiBrands $brandsApi;

	protected Sites $sites;

	protected EntityManagerDecorator $em;

	protected ProductHelper $productHelper;

	protected Settings $settings;

	protected EventDispatcher $eventDispatcher;

	protected ?array $cCategories = null;

	public function __construct(ProductsFacade           $productsFacade, ParametersService $parametersService, MallApiBrands $brandsApi, Sites $sites,
	                            ProductPromotionsService $productPromotionsService, ProductLabelsService $productLabelsService,
	                            EntityManagerDecorator   $em, ProductHelper $productHelper, ProductMapService $productMap, Settings $settings,
	                            EventDispatcher          $eventDispatcher)
	{
		$this->productsFacade           = $productsFacade;
		$this->parametersService        = $parametersService;
		$this->brandsApi                = $brandsApi;
		$this->sites                    = $sites;
		$this->productPromotionsService = $productPromotionsService;
		$this->productLabelsService     = $productLabelsService;
		$this->em                       = $em;
		$this->productHelper            = $productHelper;
		$this->productMapService        = $productMap;
		$this->settings                 = $settings;
		$this->eventDispatcher          = $eventDispatcher;
	}

	public function build(MallClient $client, EshopDao\Product $eshopProd, string $categoryId, ?array $productMap = null)
	{
		if ($productMap)
			$availability = $this->getAvailabilityAdv($eshopProd, $productMap, $client);
		else
			$availability = $this->getAvailability($eshopProd);

		$id = (string) $eshopProd->getId();
		if ($productMap) {
			if ($productMap['idForMall'])
				$id = (string) $productMap['idForMall'];
		}

		$mallProd          = new Dao\Product(
			$categoryId,
			$id,
			$eshopProd->variantName ?: $eshopProd->name,
			$eshopProd->shortDescription,
			$eshopProd->description,
			(float) $eshopProd->getBasePrice(),
			$availability,
		);
		$mallProd->eshopId = $eshopProd->getId();

		if ($productMap && $productMap['price'])
			$mallProd->price = $this->productHelper->convertPrice($client, (float) $productMap['price']);

		if ($eshopProd->getManufacturer()) {
			$brand = $this->brandsApi->findBrand($client, $eshopProd->getManufacturer()->name);

			if ($brand)
				$mallProd->brand_id = $brand;
		}

		$rateMap       = ProductHelper::$vatRateMap[$client->getCountry()] ?? null;
		$mallProd->vat = $rateMap ? $rateMap[$eshopProd->vatRate] : $eshopProd->vatRate;

		if (strlen((string) $eshopProd->getEan()))
			$mallProd->barcode = (int) $eshopProd->getEan();

		if (!$productMap || !$productMap['mallId']) {
			$mallProd->stage = Dao\Product::STAGE_LIVE;
		} else {
			$mallProd->stage = $productMap['isActive'] ? Dao\Product::STAGE_LIVE : Dao\Product::STAGE_DRAFT;
		}

		$mallProd = $this->fillBase($client, $categoryId, $mallProd, $eshopProd);

		foreach ($eshopProd->variants as $eshopVariant) {
			$pm = $this->productMapService->getMap()[$client->getCountry()][$eshopVariant->getId()] ?? $productMap;
			if ($pm)
				$av = $this->getAvailabilityAdv($eshopVariant, $pm, $client);
			else
				$av = $this->getAvailability($eshopVariant);

			$id = (string) $eshopVariant->id;
			if ($id == $mallProd->id) {
				$mallProd->id = 'P' . $mallProd->id;
			}

			$mallVariant = new Dao\Variant(
				'V' . $id,
				$eshopVariant->name,
				(float) $eshopVariant->getBasePrice(),
				$eshopVariant->shortDescription,
				$eshopVariant->description,
				$av,
			);

			if ($pm && $pm['price'])
				$mallVariant->price = $this->productHelper->convertPrice($client, (float) $pm['price']);

			$mallVariant->eshopId = $eshopVariant->getId();
			if ($eshopProd->getId() === $eshopVariant->getId())
				$mallVariant->isBase = true;

			if ($eshopVariant->variantOf == $eshopVariant->id)
				$mallVariant->priority = 2;

			if (strlen((string) $eshopVariant->getEan()))
				$mallVariant->barcode = (int) $eshopVariant->getEan();

			$mallVariant = $this->fillBase($client, $categoryId, $mallVariant, $eshopVariant);
			$mallProd->addVariant($mallVariant);
		}

		$event = new DaoEvent($mallProd, ['product' => $eshopProd, 'client' => $client]);
		$this->eventDispatcher->dispatch($event, ProductBuilder::class . '::afterBuild');

		return $mallProd;
	}

	/**
	 * @param MallClient              $client
	 * @param                         $categoryId
	 * @param Dao\Product|Dao\Variant $mallProd
	 * @param EshopDao\Product        $eshopProd
	 *
	 * @return mixed
	 */
	protected function fillBase(MallClient $client, $categoryId, $mallProd, EshopDao\Product $eshopProd)
	{
		$site = $this->sites->getSites()[$client->getEshop()];

		if ($eshopProd->getRetailPrice())
			$mallProd->rrp = (float) $eshopProd->getRetailPrice();

		$availableParams = $this->parametersService->getByCategoryId($categoryId, $client->getCountry());

		foreach ($eshopProd->getFeatures() as $feature) {
			$val = $availableParams[$feature->idFeature] ?? null;

			if ($val) {
				$mallProd->addParameter($val, $feature->value);
			}
		}

		if (isset($availableParams[Param::FEATURE_SUBCATEGORIES]) && $eshopProd->defaultCategoryId) {
			$catName = $this->getEshopCategories()[$eshopProd->defaultCategoryId]['names'][$client->getCountry()] ?? null;

			if ($catName)
				$mallProd->addParameter($availableParams[Param::FEATURE_SUBCATEGORIES], $catName);
		}

		if ($mallProd instanceof Dao\Product) {
			foreach ($eshopProd->getVariantDifferences() as $feature) {
				$val = $availableParams[$feature->idFeature] ?? null;

				if ($val)
					$mallProd->addVariableParameter($val);
			}
		}

		if ($eshopProd->getGallery()) {
			foreach ($eshopProd->getGallery()->getImages() as $img) {
				$media = new Dao\Media(($client->imagesDomain ?: $site->getCurrentDomain()->domain) . $img->getFilePath());

				if ($eshopProd->getGallery()->getCover()->id === $img->id)
					$media->main = true;

				$mallProd->addMedia($media);
			}
		}

		foreach ($this->productPromotionsService->getForProduct($eshopProd->getId())[$client->getCountry()] ?? [] as $row) {
			$entity = new Dao\Promotion((float) $this->productHelper->convertPrice($client, (float) $row['price']), $row['from'], $row['to']);
			$mallProd->addPromotion($entity);
		}

		foreach ($this->productLabelsService->getForProduct($eshopProd->getId())[$client->getCountry()] ?? [] as $mallId => $row) {
			$entity = new Dao\Label($mallId, $row['from'], $row['to']);
			$mallProd->addLabel($entity);
		}

		if ($eshopProd->getTag('freeDelivery'))
			$mallProd->free_delivery = true;

		$i = 1;
		foreach ($eshopProd->getRelated() as $group) {
			foreach ($group->getProducts() as $product) {
				$mallProd->addRecommended((string) $product->getId());

				if ($i >= 30)
					break 2;
				$i++;
			}
		}

		if ($eshopProd->isOversize)
			$mallProd->setPackageSize(Dao\Product::PACKAGE_SIZE_BIGBOX);

		return $mallProd;
	}

	public function getAvailability(EshopDao\Product $eshopProd): Dao\Availability
	{
		return new Dao\Availability(
			$eshopProd->getAvailability()->getIdent() == Availability::IN_STOCK ? Dao\Availability::ACTIVE : Dao\Availability::INACTIVE,
			$eshopProd->getQuantity(),
		);
	}

	public function getAvailabilityAdv(EshopDao\Product $eshopProd, array $productMap, MallClient $client): Dao\Availability
	{
		$disabledManufacturers = $this->settings->getDisabledManufacturers($client->getCountry()) ?? [];
		if (!$productMap['isActive']
			|| $eshopProd->getManufacturer() && in_array($eshopProd->getManufacturer()->id, $disabledManufacturers))
			return new Dao\Availability(Dao\Availability::INACTIVE, 0);

		$av = $this->getAvailability($eshopProd);

		$event = new DaoEvent($av, ['product' => $eshopProd, 'client' => $client]);
		$this->eventDispatcher->dispatch($event, ProductBuilder::class . '::getAvailabilityAdv');

		return $av;
	}

	protected function getEshopCategories(): array
	{
		if ($this->cCategories === null) {
			$this->cCategories = [];

			foreach ($this->em->createQueryBuilder()->select('c.id, IDENTITY(c.parent) as parent')
				         ->from(Category::class, 'c')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cCategories[$row['id']] = [
					'id'     => $row['id'],
					'parent' => $row['parent'],
					'names'  => [],
				];
			}

			foreach ($this->em->createQueryBuilder()->select('IDENTITY(ct.id) as id, ct.name, ct.lang')
				         ->from(CategoryTexts::class, 'ct')
				         ->getQuery()->getArrayResult() as $row) {
				if (!isset($this->cCategories[$row['id']]))
					continue;

				$this->cCategories[$row['id']]['names'][$row['lang']] = $row['name'];
			}
		}

		return $this->cCategories;
	}
}
