<?php declare(strict_types = 1);

namespace Mall\Model;

use Core\Model\Entities\EntityManagerDecorator;
use Mall\Model\Dao\MallClient;
use Mall\Model\Dao\Product\Availability;
use Mall\Model\Dao\Product\Product;
use Mall\Model\Entities\ProductForceToken;

class ProductHelper
{
	public static array $vatRateMap = [
		'sk' => [
			21 => 20,
			15 => 10,
			10 => 10,
		],
	];

	protected EntityManagerDecorator $em;

	public function __construct(EntityManagerDecorator $em)
	{
		$this->em = $em;
	}

	public static function checkAvailabilityStatus(string $status): bool
	{
		return in_array($status, [Availability::ACTIVE, Availability::INACTIVE, Availability::NOT_FOR_RESALE]);
	}

	public static function checkPackageSize(string $packageSize): bool
	{
		return in_array($packageSize, [Product::PACKAGE_SIZE_SMALLBOX, Product::PACKAGE_SIZE_BIGBOX]);
	}

	public function findForceToken($id, $country): ?ProductForceToken
	{
		return $this->em->getRepository(ProductForceToken::class)->createQueryBuilder('pft')
			->andWhere('pft.eshopProduct = :id')
			->andWhere('pft.country = :country')
			->setParameters([
				'id'      => $id,
				'country' => $country,
			])->setMaxResults(1)->getQuery()->getOneOrNullResult();
	}

	public function convertPrice(MallClient $mallClient, float $price): float
	{
		if (!$mallClient->currencyEntity)
			return $price;

		return round($price / $mallClient->currencyEntity->rate, $mallClient->currencyEntity->decimals);
	}
}
