<?php declare(strict_types = 1);

namespace Mall\Model\Services;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\Query\Expr\Join;
use Mall\Model\Entities\Category;
use Mall\Model\Entities\CategoryMap;

/**
 * @method Category get($id)
 * @method Category[] getAll()
 * @method Category getReference($id)
 */
class CategoriesService extends BaseEntityService
{
	protected $entityClass = Category::class;

	protected ?array $cAllByEshopCats = null;

	protected ?array $cAllByMallCat = null;

	public function addToMap($id, $shopId): bool
	{
		try {
			$entity = new CategoryMap(
				$this->em->getReference(Category::class, $id),
				$this->em->getReference(\EshopCatalog\Model\Entities\Category::class, $shopId),
			);
			$this->em->persist($entity);
			$this->em->flush();

			return true;
		} catch (\Exception $e) {
		}

		return false;
	}

	public function removeFromMap($id, $shopId): bool
	{
		try {
			$query = $this->em->createQueryBuilder()->delete(CategoryMap::class, 'cm')
				->where('cm.mallCategory = :mallCategory')
				->andWhere('cm.category = :category')
				->setParameters([
					'mallCategory' => $id,
					'category'     => $shopId,
				])->getQuery();
			$query->execute();

			return true;
		} catch (\Exception $e) {
			bdump($e);
		}

		return false;
	}

	public function getAllCategoriesByMall(): array
	{
		$this->preloadData();

		return $this->cAllByMallCat;
	}

	public function getAllCategoriesByEshopCategory(): array
	{
		$this->preloadData();

		return $this->cAllByEshopCats;
	}

	public function findByEshopCategory(int $id): ?array
	{
		return $this->getAllCategoriesByEshopCategory()[$id] ?? null;
	}

	public function getAvailableCategories(?string $country = null): array
	{
		$arr = [];

		$qb = $this->getEr()->createQueryBuilder('c')
			->select('c.country, c.mallId, c.title')
			->andWhere('c.isAvailable = 1');

		if ($country)
			$qb->andWhere('c.country = :country')
				->setParameter('country', $country);

		foreach ($qb->getQuery()->getArrayResult() as $row) {
			if ($country)
				$arr[$row['mallId']] = $row['title'];
			else
				$arr[$row['country']][$row['mallId']] = $row['title'];
		}

		return $arr;
	}

	protected function preloadData(): void
	{
		if ($this->cAllByEshopCats === null || $this->cAllByMallCat === null) {
			$this->cAllByMallCat   = [];
			$this->cAllByEshopCats = [];

			foreach ($this->em->getRepository(CategoryMap::class)->createQueryBuilder('cm')
				         ->select('mc.mallId, mc.country, IDENTITY(cm.category) as category')
				         ->innerJoin('cm.category', 'c', Join::WITH, 'c.isPublished = 1')
				         ->innerJoin('cm.mallCategory', 'mc')
				         ->getQuery()->getArrayResult() as $row) {
				$this->cAllByEshopCats[$row['category']][$row['country']] = $row;
				$this->cAllByMallCat[$row['country']][$row['mallId']][]   = $row;
			}
		}
	}
}
