<?php declare(strict_types = 1);

namespace Mall\Model\Services;

use Core\Model\Helpers\Strings;
use Mall\Model\Dao\MallClient;
use Mall\Model\Exceptions\ForceTokenException;
use Mall\Model\MallLogger;
use Nette\Utils\Json;

abstract class MallApiBaseService
{
	protected string $baseUrl = '';

	protected string $method = 'GET';

	public array $paging = [];

	public function sendRequest(MallClient $client, string $url = '', array $args = [], array $data = [])
	{
		if (!Strings::startsWith($this->baseUrl, 'v'))
			$this->baseUrl = 'v1/' . $this->baseUrl;

		$ch           = curl_init();
		$this->paging = [];

		if ($url && !Strings::startsWith($url, '/'))
			$url = '/' . $url;

		$curlUrl = $client->getEndPoint() . $this->baseUrl . $url . '?client_id=' . $client->getClientId();

		foreach ($args as $k => $v) {
			$curlUrl .= '&' . $k . '=' . $v;
		}
		MallLogger::log("SEND REQUEST with data - $this->method - $curlUrl", $client->getCountry());
		MallLogger::log(Json::encode($data), $client->getCountry());

		curl_setopt($ch, CURLOPT_URL, $curlUrl);
		curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
		curl_setopt($ch, CURLOPT_HEADER, false);
		curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $this->method);
		curl_setopt($ch, CURLOPT_POSTFIELDS, Json::encode($data));

		curl_setopt($ch, CURLOPT_HTTPHEADER, [
			"Content-Type: application/json",
			"Accept: application/json",
			"X-Application-Name: PSHK",
		]);

		$response = curl_exec($ch);
		$error    = curl_error($ch);

		curl_close($ch);
		MallLogger::log('GET RESPONSE ' . $response, $client->getCountry());

		if ($error) {
			MallLogger::log($error, $client->getCountry());
		} else if ($response) {
			try {
				$json = Json::decode($response, Json::FORCE_ARRAY);
			} catch (\Exception $e) {
				MallLogger::log("ERROR file:" . $e->getFile() . ' line:' . $e->getLine() . ' message' . $e->getMessage(), $client->getCountry());

				return null;
			}

			if (Strings::startsWith($this->baseUrl, 'v1/')) {

				if ($json['result']['status'] === 'OK') {
					if (isset($json['paging']))
						$this->paging = $json['paging'];

					return $json['data'] ?? true;
				} else if (isset($json['data']['data']['forceToken'])) {
					throw new ForceTokenException($json['data']['data']['forceToken']);
				}
			} else {
				if (isset($json['error_code'])) {
					MallLogger::log($json['error_code'] . ' - ' . $json['error_description'], $client->getCountry());
				} else {
					return $json;
				}
			}
		}

		return null;
	}
}
