<?php declare(strict_types = 1);

namespace Mall\Model\Services;

use Core\Model\Entities\EntityManagerDecorator;
use EshopCatalog\Model\Entities\Product;
use Mall\Model\Dao\MallClient;
use Mall\Model\Dao\Product as Dao;
use Mall\Model\Entities\ProductForceToken;
use Mall\Model\Exceptions\ForceTokenException;
use Mall\Model\MallLogger;
use Mall\Model\ProductHelper;
use Nette\Utils\Json;

class MallApiProducts extends MallApiBaseService
{
	protected string $baseUrl = 'products';

	protected EntityManagerDecorator $em;

	protected ProductHelper $productHelper;

	public function __construct(EntityManagerDecorator $em, ProductHelper $productHelper)
	{
		$this->em            = $em;
		$this->productHelper = $productHelper;
	}

	public function sendProduct(MallClient $client, Dao\Product $product, bool $update = false, ?string $forceToken = null)
	{
		$this->baseUrl = 'products';
		$this->method  = 'POST';
		$url           = '';
		if ($update) {
			$this->method = 'PUT';
			$url          = '/' . $product->id;
		}

		$args = [];

		if ($forceToken)
			$args['force_token'] = $forceToken;

		try {
			return $this->sendRequest($client, $url, $args, $product->getData());
		} catch (ForceTokenException $e) {
			$entity = $this->productHelper->findForceToken($product->eshopId, $client->getCountry());

			if ($entity) {
				$entity->reset($e->getForceToken(), $product->price);
			} else {
				$entity = new ProductForceToken($this->em->getReference(Product::class, $product->eshopId), $product->price, $e->getForceToken(), $client->getCountry());
			}
			$this->em->persist($entity);
			$this->em->flush();
		}

		return null;
	}

	public function sendAvailabilities(MallClient $client, array $data)
	{
		$this->method  = 'POST';
		$origBaseUrl   = $this->baseUrl;
		$this->baseUrl = 'availability';

		MallLogger::log(Json::encode($data), $client->getCountry() . '-availability');
		$result = $this->sendRequest($client, '', [], $data);
		if ($result)
			MallLogger::log($result, $client->getCountry() . '-availability');
		$this->baseUrl = $origBaseUrl;

		return $result;
	}

	public function deleteProduct(MallClient $client, $id)
	{
		$this->method = 'DELETE';

		$result = $this->sendRequest($client, (string) $id, [], []);

		return $result;
	}

	public function deleteVariant(MallClient $client, string $productId, string $variantId)
	{
		$this->method = 'DELETE';

		$result = $this->sendRequest($client, $productId . '/variants/' . $variantId, [], []);

		return $result;
	}

	public function getProduct(MallClient $client, string $id)
	{
		$this->method = 'GET';

		return $this->sendRequest($client, '/' . $id, [], []);
	}

	public function getVariantData(MallClient $client, string $prodId, string $variantId)
	{
		$this->method = 'GET';

		return $this->sendRequest($client, '/' . $prodId . '/variants/' . $variantId, [], []);
	}

	public function getProductsList(MallClient $client)
	{
		$this->method = 'GET';

		$currentLoop = 0;
		$data        = $this->sendRequest($client, '', ['filter' => 'basic'], []);

		while ($currentLoop < 10 && !empty($this->paging) && $this->paging['page'] < $this->paging['pages']) {
			$currentLoop++;
			$data = array_merge($data, $this->sendRequest($client, '', [
				'filter' => 'basic',
				'page'   => $this->paging['page'] + 1,
			], []));
		}

		return $data;
	}

	public function getProductVariantsList(MallClient $client, string $productId)
	{
		$this->method = 'GET';

		return $this->sendRequest($client, '/' . $productId . '/variants', ['filter' => 'basic'], []);
	}

	public function sendPrice(MallClient $client, string $productId, float $price, ?float $rrp = null)
	{
		$this->method = 'PUT';

		$data = [
			'price' => $price,
		];

		if ($rrp)
			$data['rrp'] = $rrp;

		return $this->sendRequest($client, '/' . $productId . '/pricing', [], $data);
	}

	public function sendVariantPrice(MallClient $client, string $productId, string $variantId, float $price, ?float $rrp = null)
	{
		$this->method = 'PUT';

		$data = [
			'price' => $price,
		];

		if ($rrp)
			$data['rrp'] = $rrp;

		return $this->sendRequest($client, '/' . $productId . '/variants/' . $variantId . '/pricing', [], $data);
	}
}
