<?php declare(strict_types = 1);

namespace Mall\Model\Services;

use Core\Model\Countries;
use Core\Model\Entities\EntityManagerDecorator;
use Core\Model\Event\EventDispatcher;
use Core\Model\Helpers\Strings;
use Core\Model\Notifiers\MailNotifiers\LogNotifier;
use Currency\Model\Currencies;
use Currency\Model\Entities\Currency;
use Doctrine\ORM\Query\Expr\Join;
use EshopCatalog\Model\Entities\Product;
use EshopOrders\AdminModule\Model\Statuses;
use EshopOrders\FrontModule\Model\Event\OrderEvent;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderAddress;
use EshopOrders\Model\Entities\OrderCurrency;
use EshopOrders\Model\Entities\OrderItem;
use EshopOrders\Model\Entities\OrderPayment;
use EshopOrders\Model\Entities\OrderSpedition;
use EshopOrders\Model\Entities\OrderStatus;
use EshopOrders\Model\Entities\Payment;
use Mall\Model\Dao\MallClient;
use Mall\Model\Entities\Delivery;
use Mall\Model\Entities\MallOrder;
use Nette\Utils\DateTime;
use Tracy\Debugger;
use Ulozenka\Model\UlozenkaParcels;

class OrdersFacade
{
	protected EntityManagerDecorator $em;

	protected OrdersService $ordersService;

	protected Countries $countries;

	protected MallSellersService $sellersService;

	protected DeliveriesService $deliveriesService;

	protected Statuses $statusesService;

	protected Currencies $currencies;

	protected UlozenkaParcels $ulozenkaParcels;

	protected EventDispatcher $eventDispatcher;

	protected ProductMapService $productMapService;

	public function __construct(EntityManagerDecorator $em, OrdersService $ordersService, MallSellersService $sellersService,
	                            Countries              $countries, DeliveriesService $deliveriesService, Statuses $statusesService,
	                            UlozenkaParcels        $ulozenkaParcels, Currencies $currencies, EventDispatcher $eventDispatcher,
	                            ProductMapService      $productMapService)
	{
		$this->em                = $em;
		$this->ordersService     = $ordersService;
		$this->sellersService    = $sellersService;
		$this->countries         = $countries;
		$this->deliveriesService = $deliveriesService;
		$this->statusesService   = $statusesService;
		$this->ulozenkaParcels   = $ulozenkaParcels;
		$this->currencies        = $currencies;
		$this->eventDispatcher   = $eventDispatcher;
		$this->productMapService = $productMapService;
	}

	public function addOrder(MallClient $client, array $data): ?MallOrder
	{
		$this->em->beginTransaction();
		try {
			$mallSeller = $this->sellersService->get($client->getCountry());
			$addr       = $data['address'];
			/** @var Delivery $delivery */
			$delivery = $this->deliveriesService->getExisting()[$client->getCountry()][$data['delivery_method']] ?? null;

			if (!$delivery || !$mallSeller->seller->getSites())
				return null;

			$parcel = null;
			if (isset($data['branch_id'])) {
				UlozenkaParcels::$overrideShopId = $client->ulozenka;
				$parcel                          = $this->ulozenkaParcels->getBranch($data['branch_id']);

				if (!$parcel) {
					Debugger::log($data['id'], '_mallParcelError');
					LogNotifier::toDevelopers('Mall parcel ' . $data['branch_id'] . ' for order ' . $data['id'] . ' not found', 'Mall get branch');

					return null;
				}
			}

			$order = new Order($mallSeller->seller->getSites()->first()->getSite());

			$deliverDate = DateTime::createFromFormat('Y-m-d', $data['delivery_date']);
			$order->setMessage('Doručit: ' . $deliverDate->format('j. n. Y'));
			$order->setAgreedTerms(true);

			if ($data['currency'] !== $this->currencies->getDefaultCode()) {
				/** @var Currency $currency */
				$currency                = $client->currencyEntity ?: ($this->em->getRepository(Currency::class)->findOneBy(['code' => $data['currency']]) ?? null);
				$orderCurrency           = new OrderCurrency($order, $data['currency']);
				$orderCurrency->rate     = $currency ? $currency->rate : 26;
				$orderCurrency->decimals = 2;

				$order->currency = $orderCurrency;
			}

			$name = explode(' ', $addr['name'], 2);

			$addrInv = new OrderAddress(OrderAddress::ADDRESS_INVOICE);
			$addrInv->setFirstName($name[0]);
			$addrInv->setLastName($name[1]) ?? '';
			$addrInv->setEmail('');
			$addrInv->setPhone($addr['phone']);
			$addrInv->setStreet($addr['street']);
			$addrInv->setCity($addr['city']);
			$addrInv->setPostal($addr['zip']);
			$addrInv->setCountry($this->countries->get($addr['country']));

			foreach (['company', 'ico', 'dic'] as $c)
				if (isset($addr[$c]))
					$addrInv->setCompany($addr[$c]);
			$this->em->persist($addrInv);
			$order->setAddressInvoice($addrInv);

			if (isset($data['branch_id']) && $parcel) {
				$addrDeli = new OrderAddress(OrderAddress::ADDRESS_DELIVERY);
				$addrDeli->setFirstName($name[0]);
				$addrDeli->setLastName($name[1] ?? '');
				$addrDeli->setStreet($parcel->street . ' ' . $parcel->house_number);
				$addrDeli->setPhone($addr['phone']);
				$addrDeli->setCity($parcel->town);
				$addrDeli->setPostal($parcel->zip);
				$addrDeli->setCountry($addrInv->getCountry());
				$this->em->persist($addrDeli);
				$order->setAddressDelivery($addrDeli);
			}

			/** @var Product[] $products */
			$products = [];
			foreach ($this->em->createQueryBuilder()->select('p.id, p.code1, pt.name')
				         ->from(Product::class, 'p')
				         ->leftJoin('p.productTexts', 'pt', Join::WITH, 'pt.lang = :lang')
				         ->where('p.id IN (:ids)')
				         ->setParameters([
					         'lang' => $client->getCountry(),
					         'ids'  => array_map(fn($p) => $p['id'], $data['items']),
				         ])->getQuery()->getArrayResult() as $v)
				$products[$v['id']] = $v;

			$items = [];
			foreach ($data['items'] as $item) {
				$prod     = null;
				$mallProd = $this->productMapService->getByMallId($client->getCountry(), $item['article_id']);
				if ($mallProd) {
					$prod = $mallProd->getEshopProduct();
				} else {
					$tmpId = (string) $item['id'];
					if (Strings::startsWith($tmpId, 'V'))
						$tmpId = substr($tmpId, 1);
					$existCount = $this->em->getConnection()->fetchAssociative("SELECT count(id) as count FROM eshop_catalog__product where id = ?", [(int) $tmpId])['count'] ?? 0;

					if ((int) $existCount > 0)
						$prod = $this->em->getReference(Product::class, (int) $tmpId);
				}

				$orderItem = new OrderItem($prod, $order);
				$orderItem->setQuantity($item['quantity']);
				$orderItem->setPrice($this->convertPrice($client, (float) $item['price']));
				$orderItem->setVatRate($item['vat']);
				$orderItem->setCode1($prod ? $prod->code1 : '');
				$orderItem->setMoreDataValue('commission', $item['commission']);

				$orderItem->addOrderItemText($client->getCountry());
				$orderItem->getOrderItemText($client->getCountry())->setName($item['title']);

				$this->em->persist($orderItem);
				$items[] = $orderItem;
			}
			$order->setOrderItems($items);

			// Discount maji online platby
			//		if ($data['discount']) {
			//			$orderDiscount = new OrderDiscount('', $order);
			//			$orderDiscount->setType(OrderDiscount::TYPE_FIX);
			//			$orderDiscount->setPrice((float) $data['discount']);
			//			$orderDiscount->calculateDiscount((float) $data['discount']);
			//			$this->em->persist($orderDiscount);
			//		}

			$orderSpedition = new OrderSpedition($delivery->eshopSpedition, $order);
			$orderSpedition->setPrice($this->convertPrice($client, (float) $data['delivery_price']));
			$order->setSpedition($orderSpedition);

			if ($data['cod'] > 0) {
				$payment = $this->em->getRepository(Payment::class)->findOneBy(['ident' => 'cod']) ?? null;
			} else {
				$payment       = null;
				$order->isPaid = 1;
			}

			$orderPayment = new OrderPayment($payment, $order);
			$orderPayment->setPrice($this->convertPrice($client, (float) $data['cod_price']));
			$order->setPayment($orderPayment);

			$statusCreated      = $this->statusesService->get('created');
			$orderActionCreated = new OrderStatus($order, $statusCreated);
			$order->getOrderStatuses()->add($orderActionCreated);

			$eventOrder                 = new OrderEvent($order);
			$eventOrder->addrDelivery   = isset($addrDeli) ? $addrDeli : $addrInv;
			$eventOrder->addrInvoice    = $addrInv;
			$eventOrder->orderSpedition = $orderSpedition;
			$eventOrder->orderPayment   = $orderPayment;
			$eventOrder->orderItems     = $items;
			$this->eventDispatcher->dispatch($eventOrder, 'eshopOrders.orderBeforeSave');

			$this->em->persist($order);
			$this->em->persist($orderSpedition);
			$this->em->persist($orderPayment);

			$this->em->flush();

			$mallOrder = $this->ordersService->addOrder($client->getCountry(), (string) $data['id'], $order);

			$this->em->commit();

			if ($this->ordersService->existing === null)
				$this->ordersService->getExisting();
			$this->ordersService->existing[(string) $data['id']] = $order->getId();

			Debugger::log('create order ' . $order->getId(), '_mallCreateOrder');

			return $mallOrder;
		} catch (\Exception $e) {
			Debugger::log($e->getMessage(), '_mallCreateOrderError');
			if ($this->em->getConnection()->isTransactionActive())
				$this->em->rollback();
		}

		return null;
	}

	protected function convertPrice(MallClient $client, float $price): float
	{
		if (!$client->currencyEntity)
			return $price;

		return round($price * $client->currencyEntity->rate, 0);
	}
}
