<?php declare(strict_types = 1);

namespace Mall\Model\Services;

use Core\Model\Helpers\BaseEntityService;
use Doctrine\ORM\Query\Expr\Join;
use EshopOrders\Model\Entities\CarrierHistory;
use EshopOrders\Model\Entities\Order;
use EshopOrders\Model\Entities\OrderStatus;
use Mall\Model\Entities\MallOrder;

/**
 * @method MallOrder get($id)
 * @method MallOrder[] getAll()
 * @method MallOrder getReference($id)
 */
class OrdersService extends BaseEntityService
{
	protected $entityClass = MallOrder::class;

	public ?array $existing = null;

	public function checkExist(int $mallId): bool
	{
		return isset($this->getExisting()[$mallId]);
	}

	public function getByEshopId(int $id): ?MallOrder
	{
		return $this->getEr()->createQueryBuilder('mo')
			->where('mo.order = :id')
			->setParameter('id', $id)
			->getQuery()->getOneOrNullResult();
	}

	public function getExisting(): array
	{
		if ($this->existing === null) {
			$this->existing = [];

			foreach ($this->getEr()->createQueryBuilder('mo')->select('mo.mallId as mallId, IDENTITY(mo.order) as orderId')
				         ->getQuery()->getScalarResult() as $row)
				$this->existing[$row['mallId']] = $row['orderId'];
		}

		return $this->existing;
	}

	public function addOrder(string $clientCountry, string $mallId, Order $order): MallOrder
	{
		$mallOrder          = new MallOrder($order, $mallId);
		$mallOrder->country = $clientCountry;
		$this->em->persist($mallOrder);
		$this->em->flush($mallOrder);

		return $mallOrder;
	}

	public function getDiffLastStatusFinished(): array
	{
		$mallOrders = [];

		foreach ($this->em->createQueryBuilder()->select('IDENTITY(mo.order) as orderId, mo.mallId, mo.lastStatus, IDENTITY(os.status) as eshopStatus, mo.country')
			         ->from(MallOrder::class, 'mo')
			         ->innerJoin('mo.order', 'o')
			         ->innerJoin('o.orderStatuses', 'os', Join::WITH, 'os.status = :eshopStatus')
			         ->where('mo.lastStatus IN (:statuses)')
			         ->setParameters([
				         'statuses'    => [
					         MallOrder::STATUS_CANCELED,
					         MallOrder::STATUS_RETURNED,
				         ],
				         'eshopStatus' => OrderStatus::STATUS_FINISHED,
			         ])->getQuery()->getArrayResult() as $row) {
			$mallOrders[$row['orderId']] = $row;
		}

		return $mallOrders;
	}

	public function export(MallOrder $mallOrder, ?string $trackingNumber = null, ?string $trackingUrl = null): bool
	{
		try {
			if ($trackingNumber) {
				$mallOrder->trackingNumber = $trackingNumber;
			}

			if ($trackingUrl) {
				$mallOrder->trackingUrl = $trackingUrl;
			}

			$mallOrder->export();
			$this->em->persist($mallOrder);
			$this->em->flush();

			return true;
		} catch (\Exception$e) {

		}

		return false;
	}

	public function getNotTrackedIds(): array
	{
		$arr = [];

		foreach ($this->getEr()->createQueryBuilder('mo')
			         ->select('mo.mallId as mallId, IDENTITY(mo.order) as order')
			         ->where('mo.trackingNumber IS NULL')
			         ->getQuery()->getScalarResult() as $row)
			$arr[$row['mallId']] = $row['order'];

		return $arr;
	}

	public function setTracking(int $orderId, array $data): bool
	{
		try {
			$this->em->getConnection()->executeStatement("UPDATE mall__order SET `tracking_number` = ?, `tracking_url` = ? WHERE order_id = ?", [
				$data['tracking_number'],
				$data['tracking_url'],
				$orderId,
			]);

			$ch = new CarrierHistory($this->em->getReference(Order::class, $orderId), 'mdp', $data['tracking_number']);
			$this->em->persist($ch);
			$this->em->flush($ch);

			return true;
		} catch (\Exception $e) {

		}

		return false;
	}
}
